package firebase

import (
	"context"
	"fmt"
	"log"
	"os"
	"strings"
	"sync"
	"time"

	"cloud.google.com/go/firestore"
	firebase "firebase.google.com/go"
	"firebase.google.com/go/auth"
	"github.com/gin-gonic/gin"
	"google.golang.org/api/option"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/keepalive"
	"google.golang.org/grpc/status"
)

// Firebase app
var firebaseApp *firebase.App = nil

// the firebase auth client
var authClient *auth.Client

// the firestore client
var firestoreClient *firestore.Client

// firestore user data
type FirestoreEmailProperties struct {
	lastUpdateTime int64    `firestore:"omitempty"`
	TokensRemodel  *int     `firestore:"tokens_remodel,omitempty" json:"tokens_remodel"`
	TokensPaint    *int     `firestore:"tokens_paint,omitempty" json:"tokens_paint"` // pointer because it was not initialized by most users
	TokensStarting int      `firestore:"tokens_starting,omitempty" json:"tokens_starting"`
	BannedPolicy   bool     `firestore:"banned_policy,omitempty" json:"-"`
	NSFWAllowed    bool     `firestore:"nsfw_allowed,omitempty" json:"-"`
	Refill         int      `firestore:"refill,omitempty" json:"refill,omitempty"`
	FCMTokens      []string `firestore:"fcm_tokens,omitempty" json:"fcm_tokens"`
}

// firestore email properties cache - since we only read from one location
// we can cache properties here because there is no other source of change
// on write we update this and the database
// we use emails as the primary key instead of uids, because the user can easily delete
// their account and create a new one to get new tokens
var firebaseEmailPropertiesCache map[string]*FirestoreEmailProperties = make(map[string]*FirestoreEmailProperties)
var firebaseEmailPropertiesCacheLock sync.Mutex

// Init initializes all firebase assets
func Init() {
	var err error
	dirname, err := os.UserHomeDir()
	if err != nil {
		log.Fatal(err)
	}

	var paths []string = []string{"/etc/broker/", "./", "../../", fmt.Sprintf("%s/certs/firebase/", dirname)}
	for _, s := range paths {
		fullpath := fmt.Sprintf("%sbroker-service-key.json", s)
		if _, err := os.Stat(fullpath); err == nil {
			//log.Printf("Loading key from: %s\n", fullpath)
			initialize(fullpath)
			break
		}
	}

	if firebaseApp == nil {
		log.Fatalf("Error: Failed to initialize firebase\n")
	}

	// geoDB, err = geoip2.Open("/var/lib/broker/GeoLite2-Country.mmdb")
	// if err != nil {
	// 	log.Fatal(err)
	// }

	// GetFirestoreEmailProperties("dirk.morris@gmail.com") // dirk.morris@gmail.com
	// GetFirestoreEmailProperties("dmorris@metaloft.com") // dmorris@metaloft.com

	// GetFirestoreEmailProperties("dirk.morris@gmail.com") // dirk.morris@gmail.com
	// GetFirestoreEmailProperties("dmorris@metaloft.com") // dmorris@metaloft.com
}

// UID returns uid associated with that token
func UID(idToken string) (error, string) {
	if idToken == "" {
		return fmt.Errorf("empty token"), ""
	}

	err, token := getIDToken(idToken)
	if err != nil {
		return err, ""
	}
	if token == nil {
		return fmt.Errorf("empty token"), ""
	}

	return nil, token.UID
}

// Email returns email associated with that token
func Email(idToken string) (error, string) {
	if idToken == "" {
		return fmt.Errorf("empty token"), ""
	}

	// handle custom auth tokens
	if strings.HasPrefix(idToken, "custom") {
		return nil, "custom"
	}
	if strings.HasPrefix(idToken, "reimageai") {
		return nil, "reimageai"
	}
	if strings.HasPrefix(idToken, "gravitate") {
		return nil, "gravitate"
	}
	if strings.HasPrefix(idToken, "immovlan") {
		return nil, "immovlan"
	}

	err, token := getIDToken(idToken)
	if err != nil {
		return err, ""
	}
	if token == nil {
		return fmt.Errorf("empty token"), ""
	}
	m := token.Firebase.Identities

	val, ok := m["email"]
	if !ok {
		return fmt.Errorf("no email associated with token"), ""
	}

	if x, ok := val.([]interface{}); ok {
		if len(x) < 1 {
			return fmt.Errorf("invalid email: %v", val), ""
		}
		val = x[0]
	}

	return nil, fmt.Sprintf("%v", val)
}

func AuthDebugPrint(c *gin.Context) {
	log.Printf("AUTH token: %v\n", c.GetHeader("authorization"))
	token, err := authClient.VerifyIDToken(context.Background(), c.GetHeader("authorization"))
	if err != nil {
		log.Printf("Error getting auth: %v\n", err)
		return
	}
	log.Printf("UID: %v\n", token.UID)
	log.Printf("AuthTime: %v\n", token.AuthTime)
	log.Printf("Issuer: %v\n", token.Issuer)
	log.Printf("Audience: %v\n", token.Audience)
	log.Printf("Subject: %v\n", token.Subject)
	log.Printf("Firebase: %v\n", token.Firebase)
	log.Printf("Tenant: %v\n", token.Firebase.Tenant)
	log.Printf("Identities: %v\n", token.Firebase.Identities)
	log.Printf("Claims: %v\n", token.Claims)
}

func GetBannedPolicy(email string) bool {
	if email == "" {
		return false
	}
	emailProperties := GetFirestoreEmailProperties(email)
	if emailProperties == nil {
		return false
	} else {
		return emailProperties.BannedPolicy
	}
}

func GetTokenCount(email string, name string) int {
	emailProperties := GetFirestoreEmailProperties(email)
	if emailProperties == nil {
		return 0
	}
	if name == "paint" {
		return *emailProperties.TokensPaint
	} else {
		return *emailProperties.TokensRemodel
	}
}

func DecrementTokenCount(email string, tokenName string) {
	emailProperties := GetFirestoreEmailProperties(email)
	var err error
	if tokenName == "paint" {
		*emailProperties.TokensPaint = *emailProperties.TokensPaint - 1
		_, err = firestoreClient.Doc("emails/"+email).Update(context.Background(), []firestore.Update{{Path: "tokens_paint", FieldPath: nil, Value: firestore.Increment(-1)}})
	} else {
		*emailProperties.TokensRemodel = *emailProperties.TokensRemodel - 1
		_, err = firestoreClient.Doc("emails/"+email).Update(context.Background(), []firestore.Update{{Path: "tokens_remodel", FieldPath: nil, Value: firestore.Increment(-1)}})
	}
	if err != nil {
		log.Printf("Token update error: %s\n", err)
	}
}

func AccountIncrementToken(name string, tokenName string) {
	var err error
	var fieldName = fmt.Sprintf("tokens_%s", time.Now().Format("2006_01_02"))
	_, err = firestoreClient.Doc("accounts/"+name).Update(context.Background(), []firestore.Update{{Path: fieldName, Value: firestore.Increment(1)}})
	if err != nil {
		log.Printf("Token update error: %s\n", err)
	}
}

// clears the cache for a specific uid
func ClearCache(email string) {
	firebaseEmailPropertiesCacheLock.Lock()
	delete(firebaseEmailPropertiesCache, email)
	firebaseEmailPropertiesCacheLock.Unlock()
}

// getIDToken verifies a token
func getIDToken(idToken string) (error, *auth.Token) {
	token, err := authClient.VerifyIDToken(context.Background(), idToken)
	if err != nil {
		log.Printf("Error: Failed to verify ID token: %v\n", err)
		return err, nil
	}

	return nil, token
}

func initialize(keyPath string) {
	var err error
	opt := option.WithCredentialsFile(keyPath)
	opt2 := option.WithGRPCDialOption(grpc.WithKeepaliveParams(keepalive.ClientParameters{
		Time: 2 * time.Minute,
	}))
	config := &firebase.Config{ProjectID: "reimage-ai"}

	log.Print("Initializing Firebase...")
	firebaseApp, err = firebase.NewApp(context.Background(), config, opt, opt2)
	if err != nil {
		log.Fatalf("error initializing app: %v\n", err)
	}

	log.Print("Initializing Firebase Auth...")
	authClient, err = firebaseApp.Auth(context.Background())
	if err != nil {
		log.Fatalf("error initializing auth: %v\n", err)
	}

	log.Print("Initializing Firestore...")
	firestoreClient, err = firestore.NewClient(context.Background(), "reimage-ai", option.WithCredentialsFile(keyPath))
	if err != nil {
		log.Fatalf("error initializing auth: %v\n", err)
	}
}

func GetFirestoreEmailProperties(email string) *FirestoreEmailProperties {
	var err error
	var data *firestore.DocumentSnapshot

	// check cache
	now := time.Now().UnixMilli()
	firebaseEmailPropertiesCacheLock.Lock()
	val, ok := firebaseEmailPropertiesCache[email]
	firebaseEmailPropertiesCacheLock.Unlock()
	if ok && (now-val.lastUpdateTime) < 2*60*1000 {
		return val
	}

	ctx := context.Background()

	data, err = firestoreClient.Doc("emails/" + email).Get(ctx)
	if err != nil && status.Code(err) == codes.NotFound {
		var startingTokens int = getStartingTokenCount()
		_, err = firestoreClient.Doc("emails/"+email).Create(ctx, map[string]interface{}{
			"tokens_remodel":      startingTokens,
			"tokens_paint":        startingTokens,
			"tokens_starting":     startingTokens,
			"last_tokens_awarded": time.Now(),
			"creation_date":       time.Now(),
			"refill":              0,
		})
		if err != nil && status.Code(err) == codes.AlreadyExists {
			//This may happen if we do this multiple times asynchronously
			//Or sometimes I believe the client returns it AlreadyExists ayway
			//log.Printf("Failed to create user: %v %v\n", email, err)
		} else if err != nil {
			log.Printf("Error: Failed to create user: %v %v\n", email, err)
		} else {
			log.Printf("Created new user: %v (tokens: %v)\n", email, startingTokens)
		}
		data, err = firestoreClient.Doc("emails/" + email).Get(ctx)
	}
	if err != nil {
		log.Printf("Error: Failed to fetch user: %v %v\n", email, err)
		return nil
	}
	// json, err := json.MarshalIndent(data.Data(), "", "  ")
	// fmt.Println(string(json))

	var emailProperties FirestoreEmailProperties
	if err := data.DataTo(&emailProperties); err != nil {
		log.Printf("Error: Failed to serialize emailProperties: %v\n", err)
		return nil
	}

	// initialize decent defaults (fixes old accounts)
	if emailProperties.TokensStarting == 0 {
		emailProperties.TokensStarting = getStartingTokenCount()
		_, err = firestoreClient.Doc("emails/"+email).Update(context.Background(), []firestore.Update{{Path: "tokens_starting", FieldPath: nil, Value: emailProperties.TokensStarting}})
	}
	if emailProperties.TokensPaint == nil {
		_, err = firestoreClient.Doc("emails/"+email).Update(context.Background(), []firestore.Update{{Path: "tokens_paint", FieldPath: nil, Value: emailProperties.TokensStarting}})
		emailProperties.TokensPaint = &emailProperties.TokensStarting
	}
	if emailProperties.TokensRemodel == nil {
		_, err = firestoreClient.Doc("emails/"+email).Update(context.Background(), []firestore.Update{{Path: "tokens_remodel", FieldPath: nil, Value: emailProperties.TokensStarting}})
		emailProperties.TokensRemodel = &emailProperties.TokensStarting
	}

	// add to cache
	emailProperties.lastUpdateTime = now
	firebaseEmailPropertiesCacheLock.Lock()
	firebaseEmailPropertiesCache[email] = &emailProperties
	firebaseEmailPropertiesCacheLock.Unlock()

	return &emailProperties
}

// determines the number of starting tokens
func getStartingTokenCount() int {
	return 8
}

// updateFcmTokens updates the FCM tokens for the user identified by the given email.
func UpdateFcmTokens(email string, newToken string) error {
	emailProperties := GetFirestoreEmailProperties(email)
	if emailProperties == nil {
		log.Printf("FCM Token update error: email properties not found: %s\n", email)
		return nil
	}
	if emailProperties.FCMTokens == nil {
		emailProperties.FCMTokens = make([]string, 0)
	}
	//Continue to append only if the token is not already present
	for _, token := range emailProperties.FCMTokens {
		if token == newToken {
			return nil
		}
	}

	emailProperties.FCMTokens = append(emailProperties.FCMTokens, newToken)

	// Ensure only the latest 3 tokens are kept
	if len(emailProperties.FCMTokens) > 3 {
		emailProperties.FCMTokens = emailProperties.FCMTokens[1:]
	}

	// Update the FCM tokens in the database
	_, err := firestoreClient.Doc("emails/"+email).Update(context.Background(), []firestore.Update{{Path: "fcm_tokens", FieldPath: nil, Value: emailProperties.FCMTokens}})
	if err != nil {
		log.Printf("FCM Token update error: %s\n", err)
	}

	if err != nil {
		log.Printf("Failed to update FCM tokens for user %s: %v", email, err)
		return err
	}

	//log.Printf("Successfully updated FCM tokens %s for user %s",emailProperties.FCMTokens, email)
	return nil
}
