package job

import (
	"fmt"
	"log"
	"math/rand"
	"sync"
	"time"
)

// Job holds the job/process metadata
type Job struct {
	lock sync.Mutex // must hold WriteLock when changing status info

	ID        uint64 `json:"jobID"` // unique ID of this job
	Name      string `json:"name"`  // a name for this job
	Queue     string `json:"queue"`
	StartTime int64  `json:"-"`         // time.Now().UnixMilli() at job creation
	TokenName string `json:"tokenName"` // name of the token to decrement for this job (if any)
	App       string `json:"app"`       // app that started this job
	UID       string `json:"uid"`       // uid that started this job
	Version   string `json:"version"`   // version of the app that started this job

	Inputs  map[string]string      `json:"inputs"`
	Outputs map[string]ResultFile  `json:"outputs"`
	Params  map[string]interface{} `json:"params"`

	// status information (must hold lock while accessing)
	status         string      `json:"-"`
	percent        float64     `json:"-"`
	detail         interface{} `json:"-"`
	warning        string      `json:"-"`
	lastCheckTime  int64       `json:"-"` // time.Now().UnixMilli() at the last time the status was checked
	assignedTime   int64       `json:"-"` // time.Now().UnixMilli() when job assigned to worker
	assignedWorker string      `json:"-"` // the worker assigned to complete the job
	queueTime      int64       `json:"-"` // time.Now().UnixMilli() when job assigned to a queue
	assignedQueue  string      `json:"-"` // the queue this job was put in
	completed      chan int    `json:"-"` // closed on job completion (to error or complete)
	completionTime int64       `json:"-"` // time.Now().UnixMilli() when job was completed (or errored out)
}

// ResultFile stores the metadata for an file "output" of a model
type ResultFile struct {
	Filename    string `json:"filename"`    // the localfilename
	Savename    string `json:"savename"`    // the name to save on the callers system
	ContentType string `json:"contentType"` // the http content type
}

var jobTable = make(map[uint64]*Job)
var jobTableLock sync.Mutex
var nextJobID uint64 = 0

// GetJob returns a job with the specified ID
func GetJob(id uint64) *Job {
	jobTableLock.Lock()
	defer jobTableLock.Unlock()
	return jobTable[id]
}

// CreateJob creates a new job and adds it to the job table and returns a ref
func CreateJob(name string, app string, uid string, version string) *Job {
	jobID := nextID()
	job := Job{
		ID:            jobID,
		Name:          name,
		App:           app,
		UID:           uid,
		Version:       version,
		StartTime:     time.Now().UnixMilli(),
		Inputs:        make(map[string]string),
		Outputs:       make(map[string]ResultFile),
		Params:        make(map[string]interface{}),
		status:        "queued",
		lastCheckTime: time.Now().UnixMilli(),
		completed:     make(chan int),
	}
	jobTableLock.Lock()
	jobTable[jobID] = &job
	jobTableLock.Unlock()

	return &job
}

// FullName returns an official name for the job for better printing and debugging
func (j *Job) FullName() string {
	return fmt.Sprintf("[%d,%-9.9s]", j.ID, j.Name)
}

// UpdateStatus changes the status information
func (j *Job) UpdateStatus(status string, detail interface{}, warning string, percent float64) {
	j.lock.Lock()
	defer j.lock.Unlock()
	j.percent = percent
	j.detail = detail
	j.warning = warning
	if j.status != status {
		log.Printf("Job %s: %15s -> %.15s\n", j.FullName(), j.status, status)
		if status == "error" {
			log.Printf("Job %s: Error: %s [%s]\n", j.FullName(), j.detail, j.assignedWorker)
		}

		j.status = status
		if status == "complete" || status == "error" {
			if j.completed != nil {
				close(j.completed)
				j.completed = nil
				j.completionTime = time.Now().UnixMilli()
			}
		}
	}
}

// AssignWorker assign a job to a worker
func (j *Job) AssignWorker(workerName string) {
	j.lock.Lock()
	defer j.lock.Unlock()
	j.assignedTime = time.Now().UnixMilli()
	j.assignedWorker = workerName
}

// AssignQueue assign a job to a wait queue
func (j *Job) AssignQueue(queueName string) {
	j.lock.Lock()
	defer j.lock.Unlock()
	j.queueTime = time.Now().UnixMilli()
	j.assignedQueue = queueName
}

// Completed returns true if the job is completed (either "complete" or "error")
func (j *Job) Completed() bool {
	j.lock.Lock()
	complete := (j.completed == nil)
	status := j.status
	j.lock.Unlock()

	if complete && (status != "complete" && status != "error") {
		log.Printf("Constraint failed: Job %d complete but status is %s\n", j.ID, j.status)
	}
	return complete
}

// WaitForCompletion waits until the timeout
func (j *Job) WaitForCompletion(timeoutSec float64) {
	j.lock.Lock()
	c := j.completed
	j.lock.Unlock()
	if c == nil {
		return
	}
	select {
	case <-c:
		return
	case <-time.After(time.Duration(timeoutSec) * time.Second):
		return
	}
}

// GetStatusJson returns a json obj representing the status of the job
func (j *Job) GetStatusJson() map[string]interface{} {
	var jmap = make(map[string]interface{})
	j.lock.Lock()
	j.lastCheckTime = time.Now().UnixMilli()
	jmap["jobID"] = j.ID
	jmap["status"] = j.status
	if j.percent != 0.0 {
		jmap["percent"] = j.percent
	}
	if j.detail != "" {
		jmap["detail"] = j.detail
	}
	if j.warning != "" {
		jmap["warning"] = j.warning
	}
	j.lock.Unlock()

	return jmap
}

// GetStatus returns just the status string
func (j *Job) GetStatus() string {
	j.lock.Lock()
	defer j.lock.Unlock()
	return j.status
}

// GetDetail returns just the detail string
func (j *Job) GetDetail() interface{} {
	j.lock.Lock()
	defer j.lock.Unlock()
	return j.detail
}

// GetPercent returns just the percent string
func (j *Job) GetPercent() float64 {
	j.lock.Lock()
	defer j.lock.Unlock()
	return j.percent
}

// GetAssignedQueue returns just the assignedQueue string
func (j *Job) GetAssignedQueue() string {
	j.lock.Lock()
	defer j.lock.Unlock()
	return j.assignedQueue
}

// GetQueueTime returns the time this job was assigned to a queue
func (j *Job) GetQueueTime() int64 {
	j.lock.Lock()
	defer j.lock.Unlock()
	return j.queueTime
}

// GetLastCheckTime returns the last time this job was checked by the client
func (j *Job) GetLastCheckTime() int64 {
	j.lock.Lock()
	defer j.lock.Unlock()
	return j.lastCheckTime
}

// GetAssignedTime returns the time this job was assigned (or 0)
func (j *Job) GetAssignedTime() int64 {
	j.lock.Lock()
	defer j.lock.Unlock()
	return j.assignedTime
}

// GetAssignedWorker returns the assigned worker (or "" is none)
func (j *Job) GetAssignedWorker() string {
	j.lock.Lock()
	defer j.lock.Unlock()
	return j.assignedWorker
}

// JobCleaner goes through jobs and just removes old completed jobs from jobTable
func JobCleaner() {
	ticker := time.NewTicker(60 * time.Second)
	for {
		select {
		case <-ticker.C:
			var cutoffMin int64 = 10
			now := time.Now().UnixMilli()

			jobTableLock.Lock()
			for id, job := range jobTable {
				job.lock.Lock()
				if job.completionTime != 0 && now-job.completionTime > 1000*60*cutoffMin {
					//log.Printf("Deleting old completed job: %d %s (%d seconds old)\n", id, job.status, (now-job.completionTime)/1000)
					delete(jobTable, id)
				}
				job.lock.Unlock()
			}
			jobTableLock.Unlock()

		}
	}
}

// nextID returns the next unique available job ID
func nextID() uint64 {
	jobTableLock.Lock()
	defer jobTableLock.Unlock()
	if nextJobID == 0 {
		nextJobID = uint64(time.Now().UnixMilli() * 10)
	}
	jobID := nextJobID
	nextJobID = nextJobID + (rand.Uint64() % 1000000000)
	//nextJobID = nextJobID + 1
	return jobID
}
