"""stable diffusion service provider"""

import os

os.environ["NLTK_DATA"] = "mldata/nltk_data"
import sys
import torch
import tempfile
import traceback
import numpy as np
import cv2
from transformers import AutoImageProcessor , AutoModelForDepthEstimation
from controlnet_aux import (
    MLSDdetector,
)
import re
import torchvision.transforms as transforms
from PIL import Image
from torchvision.transforms.functional import pil_to_tensor, to_pil_image
import models
from .util import tmpname



def mlsd(image=None):
    if image is None:
        raise Exception("image required")
    mlsd_image_path = tmpname(suffix=".png", prefix="mlsd-")
    img = conditioner_mlsd(image)
    img.save(mlsd_image_path)
    return mlsd_image_path


def canny(image=None):
    if image is None:
        raise Exception("image required")
    canny_image_path = tmpname(suffix=".png", prefix="canny-")
    img = conditioner_canny(image)
    img.save(canny_image_path)
    return canny_image_path


def depth(image=None, colormap=None):
    if image is None:
        raise Exception("image required")
    depth_image_path = tmpname(suffix=".png", prefix="depth-")
    img = conditioner_depth(image)
    img.save(depth_image_path)

    # do nothing if colormap = grayscale r None
    if colormap is None or colormap == "grayscale":
        return depth_image_path

    try:
        colormap = re.sub(r"\W+", "", colormap)
        im = cv2.imread(depth_image_path)
        # pylint: disable=eval-used
        im = cv2.applyColorMap(im, eval(f"cv2.COLORMAP_{colormap.upper()}"))
        cv2.imwrite(depth_image_path, im)
    except:
        # just ignore return the un-colormaped image
        traceback.print_exc(file=sys.stdout)
    return depth_image_path


def conditioner_pack(image):
    """This is just an API call to return 5 conditioned images used by remodel AI"""
    results = {}
    results["image-0"] = depth(image)
    results["image-1"] = depth(image, colormap="magma")
    results["image-2"] = mlsd(image)
    results["image-3"] = canny(image)
    results["image-4"] = hed(image)
    return results


def conditioner_mlsd(image_path):
    # https://huggingface.co/lllyasviel/sd-controlnet-mlsd
    height, width, _ = cv2.imread(image_path).shape
    img = Image.open(image_path).convert("RGB")
    res = 1024
    mlsd_model = MLSDdetector.from_pretrained("lllyasviel/ControlNet")
    img = mlsd_model(img, detect_resolution=res, image_resolution=res)
    img = img.resize((width, height))
    return img


def conditioner_canny(image_path):
    # Use Edge Drawing which gives better canny-like edges
    image = cv2.imread(image_path, cv2.IMREAD_GRAYSCALE)
    ed = cv2.ximgproc.createEdgeDrawing()
    params = cv2.ximgproc.EdgeDrawing.Params()
    params.PFmode = True
    ed.setParams(params)
    edges = ed.detectEdges(image)
    edge_map = ed.getEdgeImage(edges)
    return Image.fromarray(edge_map)


def conditioner_depth(image_path):
    # This code is pulled from
    # https://huggingface.co/docs/transformers/main/en/model_doc/depth_anything_v2
    height, width, _ = cv2.imread(image_path).shape
    image = Image.open(image_path).convert("RGB")
    depth_processor =  AutoImageProcessor.from_pretrained("depth-anything/Depth-Anything-V2-Base-hf", local_files_only=False)
    depth_model = AutoModelForDepthEstimation.from_pretrained("depth-anything/Depth-Anything-V2-Base-hf", local_files_only=False).to("cuda")
    inputs = depth_processor(images=image, return_tensors="pt").to("cuda")
    with torch.no_grad():
        outputs = depth_model(**inputs)
    post_processed_output = depth_processor.post_process_depth_estimation(
        outputs,
        target_sizes=[(image.height, image.width)],
    )
    predicted_depth = post_processed_output[0]["predicted_depth"]
    depth = (predicted_depth - predicted_depth.min()) / (predicted_depth.max() - predicted_depth.min())
    depth = depth.detach().cpu().numpy() * 255
    depth = Image.fromarray(depth.astype("uint8"))
    return depth

