"""Configuration loader for standardized workflow system"""
# pylint: disable=too-many-branches
# pylint: disable=broad-exception-raised
# pylint: disable=broad-exception-caught
import os
import sys
import io
import time
import json
import pprint
from typing import Dict, Any
import traceback
from pathlib import Path
import yaml
from PIL import Image
from comfy_api_simplified import ComfyWorkflowWrapper
from comfy_api import CustomComfyApiWrapper

# pylint: disable=unused-import
# pylint: disable=wrong-import-order
import random # used in eval expressions

CONFIG_DIR = os.path.join(os.path.dirname(__file__), 'configs')

AVAILABLE_CONFIGS = sorted([f for f in os.listdir(CONFIG_DIR) if f.endswith('.yaml')])

CONFIG_FILENAME_TO_CONFIG_CACHE = {}


def run_inputs(inputs: Dict[str, Any],):
    """Find the appropriate config for the given inputs and run it"""
    config = _find_config_for_inputs(inputs)
    if config is None:
        raise Exception(f"No configuration found for inputs: {inputs}")
    return _run_config_with_inputs(config, inputs)

def _find_config_for_inputs(inputs: Dict[str, Any]) -> Dict[str, Any]:
    """
    Find the appropriate workflow for a given set of inputs

    Args:
        inputs: A dict of all the user inputs and their values

    Returns:
        The appropriate config filename (or None if no match found)
    """
    for fname, cfg in CONFIG_FILENAME_TO_CONFIG_CACHE.items():
        when = cfg.get('basic_config').get('when')
        # pylint: disable=eval-used
        try:
            value = eval(when, {}, inputs)
        # pylint: disable=bare-except
        except:
            value = False
        if value is True:
            print(f"Matched config {fname}: \"{when}\" is {value}", flush=True)
            return cfg
        # print(f"Skipped config {fname}: \"{when}\" is {value}", flush=True)

    print(f"No config found for inputs: {inputs}", flush=True)
    return None

def _run_config_with_inputs(config: Dict[str, Any], inputs: Dict[str, Any]):
    """run the specified configuration with the specified inputs"""
    if config is None:
        raise Exception(f"Invalid config: {config}")

    _verify_inputs_with_config(config, inputs)

    # Remove special param maintain_size from inputs to config
    maintain_size = inputs.pop('maintain_size', False) in [True, "true", "True", "1"] and inputs.get('image') is not None
    input_width, input_height = 0, 0
    if maintain_size:
        input_width, input_height = Image.open(inputs.get('image')).size
        print(f"Image dimensions: Width={input_width}, Height={input_height}")

    # Initialize ComfyUI API
    api = CustomComfyApiWrapper("http://127.0.0.1:8188/")
    workflow_path = os.path.dirname(__file__) + "/workflows/" + config['basic_config']['workflow']
    workflow = ComfyWorkflowWrapper(workflow_path)

    try:
        # Apply configuration to workflow
        print(f"Configuring workflow {config['basic_config']['workflow']} using config {config['filename']} ...", flush=True)
        _apply_config_to_workflow(workflow, api, config, inputs)
    except Exception as exc:
        traceback.print_exc(file=sys.stdout)
        raise Exception(f"Failed to load configuration: {exc}") from exc

    print("================= RUNNING WORKFLOW =================")
    print(f"Config:   {config.get('filename')}")
    print(f"Workflow: {config.get('basic_config').get('workflow')}")
    print("Inputs: ")
    print(f" {pprint.pformat(inputs, compact=False)[1:-1]}")
    print("================= RUNNING WORKFLOW =================")

    # Execute workflow
    time1 = time.time()
    try:
        result = api.queue_and_wait_results(workflow)
    except Exception as exc:
        print(f"Failed to execute workflow: {exc}")
        raise exc
    time2 = time.time()
    print(f"Workflow completed. ({time2-time1:0.2f}s)", flush=True)

    outputs = _fetch_outputs(config, result, api, workflow, maintain_size, input_width, input_height)

    try:
        api.free()
    except Exception as exc:
        traceback.print_exc(file=sys.stdout)

    time3 = time.time()
    print("================ COMPLETION SUMMARY ================")
    print(f"Config:   {config.get('filename')}")
    print(f"Workflow: {config.get('basic_config').get('workflow')}")
    print("Inputs: ")
    print(f" {pprint.pformat(inputs, compact=False)[1:-1]}")
    print("Outputs: ")
    print(f" {pprint.pformat(outputs, compact=False)[1:-1]}")
    print(f"Total Time: {time3-time1:0.2f}s")
    print("================ COMPLETION SUMMARY ================", flush=True)

    return outputs

def _fetch_outputs(config: Dict[str, Any], result, api: CustomComfyApiWrapper, workflow: ComfyWorkflowWrapper, maintain_size=False, input_width=0, input_height=0):
    """fetch the results from the workflow output"""
    outputs = {}
    for output_name, output_config in config["outputs"].items():
        output_type = output_config.get('type')
        output_node_title = output_config.get('node')
        output_node_field = output_config.get('field')
        output_node_id = None
        if output_node_title is not None:
            output_node_id = workflow.get_node_id(output_node_title)

        if output_type == "images":
            images = result["outputs"][output_node_id][output_node_field]
            for x, image in enumerate(images):
                image_data = api.get_image(image["filename"], image["subfolder"], image["type"])
                name = image["filename"].replace("_.", ".") # fix annoying suffix from comfy
                fname = f"/tmp/{name}"
                if maintain_size and 0 not in [input_width, input_height]:
                    output = Image.open(io.BytesIO(image_data))
                    output_width, output_height = output.size
                    if (output_width, output_height) != (input_width, input_height):
                        resized_image = output.resize((input_width, input_height), resample=Image.LANCZOS)
                        print(f"Resized {output_name}-{x} to {input_width}x{input_height}")
                        resized_image.save(fname)
                        print(f"Saved {fname}", flush=True)
                    else:
                        with open(fname, "wb+") as outputfile:
                            outputfile.write(image_data)
                            print(f"Saved {fname}", flush=True)
                else:
                    with open(fname, "wb+") as outputfile:
                        outputfile.write(image_data)
                        print(f"Saved {fname}", flush=True)
                outputs[f"{output_name}-{x}"] = fname

        elif output_type == "json_string":
            value = result["outputs"][output_node_id][output_node_field]
            if isinstance(value, list) and len(value) == 1:
                value = value[0]
            json_result = json.loads(value)
            outputs[output_name] = json_result

        elif output_type == "text":
            value = result["outputs"][output_node_id][output_node_field]
            # some nodes return a list with a single value
            if isinstance(value, list) and len(value) == 1:
                value = value[0]
            outputs[output_name] = value

        elif output_type == "eval":
            # pylint: disable=eval-used
            # pylint: disable=bare-except
            try:
                eval_expr = output_config['eval']
                outputs[output_name] = eval(eval_expr, globals(), outputs)
            except:
                traceback.print_exc(file=sys.stdout)

    return outputs

def _verify_inputs_with_config(config: Dict[str, Any], inputs: Dict[str, Any]):
    # Verify inputs by looping through actual inputs
    for input_name, input_value in inputs.items():
        input_conf = config['inputs'].get(input_name)
        if input_conf is None:
            # ignored input
            continue
        if input_conf.get('type') is not None:
            if input_conf.get('type') == 'filepath':
                # check filepath type manually as its not a native python type
                if not isinstance(input_value, Path):
                    raise ValueError(f"Invalid input type {input_name}: {input_value} - got `{type(input_value).__name__}` expected `{input_conf.get('type')}`")
                if not input_value.exists():
                    raise ValueError(f"File not found: {input_value}")
            elif type(input_value).__name__ != input_conf.get('type'):
                raise ValueError(f"Invalid input type {input_name}: {input_value} - got `{type(input_value).__name__}` expected `{input_conf.get('type')}`")

    # Verify inputs by looping through input configs
    if not isinstance(config['inputs'], dict):
        raise ValueError("Invalid inputs")
    for input_name, input_conf in config['inputs'].items():
        if not isinstance(input_name, str) or not isinstance(input_conf, dict):
            raise ValueError(f"Invalid input {input_name}")
        for required_field in ['required', 'type']:
            if required_field not in input_conf:
                raise ValueError(f"Missing required field: {input_name}.{required_field}")
        if input_name not in inputs and input_conf.get('required') is True:
            raise ValueError(f"Missing required input: {input_name}")

def _apply_config_to_workflow(workflow: ComfyWorkflowWrapper, api: CustomComfyApiWrapper, config: Dict[str, Any], inputs: Dict[str, Any]) -> None:
    """
    Apply configuration to a ComfyUI workflow

    Args:
        workflow: ComfyWorkflowWrapper instance
        config: Configuration dictionary from YAML
    """
    # Upload any files
    for input_name, input_value in inputs.items():
        input_conf = config['inputs'].get(input_name)
        if input_conf is None:
            # ignored input
            continue
        if input_conf.get('type') == 'filepath':
            print(f"Uploading {input_value} ...")
            image_metadata = api.upload_image(input_value, subfolder='uploads')
            inputs[input_name] = f"{image_metadata['subfolder']}/{image_metadata['name']}"

    # Set defaults for any unspecified inputs:
    for input_name, input_conf in config['inputs'].items():
        if 'default' in input_conf and input_name not in inputs:
            inputs[input_name] = input_conf['default']

    # Apply node configurations
    if 'changes' in config:
        if not isinstance(config['changes'], dict):
            raise ValueError("Invalid changes list")
        for field, value in config.get('changes').items():
            _apply_change_to_workflow(workflow, field, value, inputs)

    # Apply Inputs changes
    for input_name, input_conf in config['inputs'].items():
        if input_name in inputs and 'changes' in input_conf:
            if not isinstance(input_conf['changes'], dict):
                raise ValueError(f"Invalid change list for user input {input_name}")
            for field, value in input_conf['changes'].items():
                _apply_change_to_workflow(workflow, field, value, inputs)

    # Apply conditional changes
    if 'conditional_changes' in config:
        for conditional_change in config.get('conditional_changes'):
            when = conditional_change.get('when')
            changes = conditional_change.get('changes')
            if None in [when, changes]:
                raise ValueError("Invalid conditional change - missing field")
            if not isinstance(when, str) or not isinstance(changes, dict):
                raise ValueError("Invalid conditional change")
            # pylint: disable=eval-used
            try:
                value = eval(when, {}, inputs)
            # pylint: disable=bare-except
            except:
                value = False
            if value is True:
                print(f"Matched conditional \"{when}\" is {value}", flush=True)
                for field, value in changes.items():
                    _apply_change_to_workflow(workflow, field, value, inputs)
            else:
                print(f"Skipped conditional \"{when}\" is {value}", flush=True)

def _apply_change_to_workflow(workflow: ComfyWorkflowWrapper, field: str, value: str, inputs: Dict[str, Any]) -> None: # 
    """
    Apply a change to a workflow

    Returns:
        None
    """
    if not isinstance(field, str) or not isinstance(inputs, dict):
        raise ValueError(f"Invalid change {field}")

    try:
        node, field = field.rsplit(".", 1)

        # Store the current value of the node param in "value"
        # This is so the config has access to the current value, allowing for the current value to be modified
        evalinputs = inputs.copy()
        evalinputs['value'] = workflow.get_node_param(node, field)

        fstrvalue = _fstr(value, evalinputs)
        workflow.set_node_param(node, field, fstrvalue)
        printvalue = fstrvalue.replace('\n', ' ')
        print(f"Applied user input: {node}.{field} = {printvalue}", flush=True)
    except Exception as exc:
        # pylint: disable=broad-exception-raised
        raise Exception(f"Failed to apply change {field} {fstrvalue}") from exc

def _fstr(template, inputs):
    """evaluate f-string templates in python"""
    # pylint: disable=eval-used
    # pylint: disable=bare-except
    try:
        # pylint: disable=unused-import
        return eval(f'f"""{template}"""', globals(), inputs)
    except:
        traceback.print_exc(file=sys.stdout)
        return sys.exc_info()[0]

def _load_config(config_filename: str) -> Dict[str, Any]:
    """
    Load configuration for given filename

    Args:
        config_filename: The name of the config to load

    Returns:
        Configuration dictionary

    Raises:
        FileNotFoundError: If config file doesn't exist
        ValueError: If config validation fails
    """
    config_path = os.path.join(CONFIG_DIR, config_filename)

    if not os.path.exists(config_path):
        raise FileNotFoundError(f"Configuration file not found: {config_path}")

    with open(config_path, 'r', encoding="utf-8") as fff:
        config = yaml.safe_load(fff)

    # add the filename
    config['filename'] = config_filename

    # Validate required fields
    if 'basic_config' not in config:
        raise ValueError(f"Missing 'basic_config' in {config_filename}")
    if 'workflow' not in config['basic_config']:
        raise ValueError(f"Missing 'workflow' in basic_config of {config_filename}")
    if 'outputs' not in config:
        raise ValueError(f"Missing 'outputs' in basic_config of {config_filename}")
    if 'inputs' not in config:
        raise ValueError(f"Missing 'inputs' in basic_config of {config_filename}")

    return config


for filename in AVAILABLE_CONFIGS:
    CONFIG_FILENAME_TO_CONFIG_CACHE[filename] = _load_config(filename)
