"""Test cases for comfy-service"""
# pylint: disable=bare-except
# pylint: disable=broad-exception-raised
import os
import pprint
import shutil
import re
import traceback
import sys
from pathlib import Path
from functools import partial
import config_manager

def filepath(value):
    """
    Checks to see if a str describes a file path and checks a few locations for convenience
    
    If so it returns the full file path.
    If no file was found returns None
    """
    # if not isinstance(value, Path):
    #     raise ValueError(f"Invalid path object {value}")
    for possible_file_path in [
            value,
            f"../{value}",
            f"{os.path.dirname(__file__)}/{value}",
            f"{os.path.dirname(__file__)}/../{value}",
            f"{os.path.dirname(__file__)}/../cache/{value}"
    ]:
        if os.path.isfile(possible_file_path):
            return Path(possible_file_path)
    return None

class TestCase:
    '''
    An individual test case

    name - a name for the test case
    inputs - a set of inputs
    success_fn - a success/failure evaluator for this test
    '''
    def __init__(
            self,
            inputs,
            success_fn=None,
    ):
        self.inputs = inputs
        self.success_fn = lambda a: True
        if success_fn is None:
            self.success_fn = lambda a: True
        else:
            self.success_fn = success_fn

        # Remove all None values from the inputs
        # In these tests "None" is a way to test unspecified
        # There is no way for a user to specify the literal None value from the calling the API
        for k in list(self.inputs.keys()):
            if self.inputs[k] is None:
                self.inputs.pop(k)


    def savename(self, count, outputkey, extension):
        '''get the a decent savename prefix'''
        copy = self.inputs.copy()
        copy.pop('op', None)
        if 'prompt' in copy:
            copy['prompt'] = 'prompt'
        if 'seed' in copy:
            copy['seed'] = f"{copy['seed']:04d}"

        description = ""
        for name,val in copy.items():
            if "image" in name:
                continue
            strval = str(val)
            strval = re.sub(r"\s+", "", strval)
            strval = strval if len(strval) <= 10 else f"{strval:10}"
            description = f"{name}:{strval}-{description}"
        description = description[:-1]
        image_base = os.path.basename(self.inputs.get('image')).split(".")[0]
        return f"output-{count:04d}-{self.inputs.get('op')}-{image_base}-{description}.{outputkey}{extension}"


    def run(self, savedir, count, extra_success_fn=None):
        '''run a test case'''
        print("Inputs:")
        print(f" {pprint.pformat(self.inputs, compact=False)[1:-1]}")
        outputs = config_manager.run_inputs(self.inputs)
        # Save outputs
        for outputkey, outputvalue in outputs.items():
            if not isinstance(outputvalue, str):
                continue
            if not os.path.exists(outputvalue):
                continue
            if os.stat(outputvalue).st_size == 0:
                continue # skip if empty
            outputname = savedir + "/" + self.savename(count, outputkey, os.path.splitext(outputvalue)[1])
            print(f"Saving {outputkey} to {outputname}")
            shutil.copy(outputvalue, outputname)

        # Validate output
        result = self.success_fn(outputs)
        if result is True:
            result = extra_success_fn(outputs)
        return result, outputs


class TestCaseSet:
    '''
    Test Case defines a set of inputs and eval to check that the evaluated correctly
    
    success_fn - a success fn to check all test case results
    '''
    # pylint: disable=too-many-arguments
    def __init__(
            self,
            name,
            success_fn=None,
    ):
        self.name = name
        self.test_cases = []
        if success_fn is None:
            self.success_fn = lambda a: True
        else:
            self.success_fn = success_fn

    def append(self, case):
        '''append a inputs to the test set'''
        self.test_cases.append(case)

    def run(self, buffer_output, quick=False):
        '''run the set of test cases'''
        if len(self.test_cases) == 0:
            print(f"Skipping {self.name} (no test cases)")
            return True
        print(f"Running {self.name} (cases: {len(self.test_cases)})...")
        savedir = f"./outputs/outputs-{self.name}"
        os.makedirs(savedir, exist_ok=True)

        for i, testcase in enumerate(self.test_cases):
            testname = f"{self.name}-{i:04d}"
            print(f"==== Test {self.name} {testname} RUN  ====")

            buffer_output.seek(0)
            buffer_output.truncate(0)

            try:
                result, outputs = testcase.run(savedir, i, extra_success_fn=self.success_fn)
            except:
                traceback.print_exc(file=sys.stdout)
                result, outputs = False, {}

            if not result:
                print(f"==== Test {self.name} {testname} FAIL ====")
                print("Inputs:")
                print(f" {pprint.pformat(testcase.inputs, compact=False)[1:-1]}")
                print("Outputs:")
                print(f" {pprint.pformat(outputs, compact=False)[1:-1]}")
                print(f"Result: {result}")
                print(f"Comfy Output: \n{buffer_output.getvalue()}")
                return False, i+1 # immediately exit
            print(f"==== Test {self.name} {testname} PASS ====\n\n")
            if quick:
                return True, i+1
        # All tests passed
        return True, len(self.test_cases)

    @staticmethod
    def create_file_list(name, namefilter=None):
        '''creates an iterator for the files in a directory'''
        found_dir = None

        if os.path.isfile(os.path.join(os.path.dirname(__file__),f"../{name}")):
            return [os.path.join(os.path.dirname(__file__),f"../{name}")]
        if os.path.isfile(os.path.join(os.path.dirname(__file__),name)):
            return [os.path.join(os.path.dirname(__file__),name)]

        if os.path.exists(os.path.join(os.path.dirname(__file__),f"../{name}")):
            found_dir = os.path.join(os.path.dirname(__file__),f"../{name}")
        if os.path.exists(os.path.join(os.path.dirname(__file__),name)):
            found_dir = os.path.join(os.path.dirname(__file__),name)
        if found_dir is None:
            raise Exception(f"Unable to locate files directory: {name}")

        file_list = sorted(os.listdir(found_dir))
        file_list = [file for file in file_list if os.path.isfile(os.path.join(found_dir, file))]
        if namefilter is not None:
            file_list = [file for file in file_list if namefilter in file or namefilter.replace(" ","_") in file]
        file_list = [os.path.normpath(f"{found_dir}/{f}") for f in file_list]
        file_list = [Path(f) for f in file_list]
        return file_list


def _get_room_type_from_filename(filename):
    '''deduces the room_type from filename'''
    room_types = ["hall", "kitchen", "living room", "bathroom", "bedroom", "dining room", "office", "garage", "laundry room"]
    for room_type in room_types:
        if room_type in str(filename):
            return room_type
        if room_type.replace(' ','_') in str(filename):
            return room_type
        if room_type.replace(' ','-') in str(filename):
            return room_type
    raise Exception(f"Unknown room_type for: {filename}")


def staging_sets(seed_list):
    '''create the test suite'''
    sets = []

    # Test staging with presets
    suite = TestCaseSet(
        "staging-preset",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    presets = ["bohemian", "cabin", "farmhouse", "industrial", "modern", "minimalist", "scandinavian"]
    for seed in seed_list:
        for preset in presets:
            for image in TestCaseSet.create_file_list("test-images/staging/empty-rooms/"):
                suite.append(TestCase(
                    {
                    "op": "staging",
                    "seed": seed,
                    "preset": preset,
                    "image": image,
                    "room_type": _get_room_type_from_filename(image),
                }))
    sets.append(suite)

    # Test staging with style image
    suite = TestCaseSet(
        "staging-style",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    room_types = ["hall", "kitchen", "living room", "bathroom", "bedroom", "dining room", "office", "garage"]
    for room_type in room_types:
        for seed in seed_list:
            for image in TestCaseSet.create_file_list("test-images/staging/empty-rooms/"):
                for style_image in TestCaseSet.create_file_list("test-images/staging/style-images/", namefilter=room_type):
                    suite.append(TestCase({
                        "op": "staging",
                        "seed": seed,
                        "style_image": style_image,
                        "image": image,
                        "room_type": room_type,
                    }))
    sets.append(suite)

    # Test staging warning about already staged rooms
    suite = TestCaseSet(
        "staging-warning",
        success_fn=lambda outputs : outputs.get('warning') is not None and outputs.get('warning') != "",
    )
    suite.append(TestCase({
        "op": "staging",
        "preset": "modern",
        "seed": 0,
        "room_type": "living room",
        "image": filepath("test-images/living-room/05-living-room.jpg"),
    }))
    sets.append(suite)

    return sets


def unstaging_sets(seed_list):
    '''create the test suite'''
    sets = []

    suite = TestCaseSet(
        "unstaging",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    for seed in seed_list:
        for image in TestCaseSet.create_file_list("test-images/unstaging/"):
            suite.append(TestCase({
                "op": "unstaging",
                "strength": 0.9,
                "seed": seed,
                "image": image,
            }))
    sets.append(suite)

    suite = TestCaseSet(
        "unstaging-prompt",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    sample_inputs = [
        (filepath("test-images/unstaging/01-bedroom-orig.jpg"), "remove the bed"),
        (filepath("test-images/unstaging/01-bedroom-orig.jpg"), "remove the dressing table"),
        (filepath("test-images/unstaging/01-bedroom-orig.jpg"), "remove the chair"),
        (filepath("test-images/unstaging/04-living-room-orig.jpg"), "remove the sofa"),
        (filepath("test-images/unstaging/04-living-room-orig.jpg"), "remove the cabinet"),
        (filepath("test-images/unstaging/dining-room-2.png"), "remove the table and chairs"),
        (filepath("test-images/unstaging/01-office-orig.jpg"), "remove 2 chairs"),
        (filepath("test-images/unstaging/02-kitchen-square.jpg"), "remove the fridge"),
    ]
    for seed in seed_list:
        for (image_path, prompt) in sample_inputs:
            suite.append(TestCase({
                "op": "unstaging-user-prompt",
                "image": image_path,
                "seed": seed,
                "prompt": prompt,
                "denoise_strength": 1.0,
            }))
    sets.append(suite)

    return sets


def interior_remodel_sets(seed_list):
    '''create the test suite'''
    sets = []

    suite = TestCaseSet(
        "interior-remodel-masked",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    masked_testcases = [
        (filepath("test-images/interior-remodel/bedroom_1.png"), filepath("test-images/interior-remodel/bedroom_1.mask.png"), "Japandi"),
        (filepath("test-images/interior-remodel/bedroom_2.png"), filepath("test-images/interior-remodel/bedroom_2.mask.png"), "Japandi"),
        (filepath("test-images/interior-remodel/dining_room_1.png"), filepath("test-images/interior-remodel/dining_room_1.mask.png"), "Modern Southwestern"),
        (filepath("test-images/interior-remodel/kitchen_1.png"), filepath("test-images/interior-remodel/kitchen_1.mask.png"), "Modern Southwestern"),
        (filepath("test-images/interior-remodel/kitchen_2.png"), filepath("test-images/interior-remodel/kitchen_2.mask.png"), "Scandinavian"),
        (filepath("test-images/interior-remodel/living_room_1.png"), filepath("test-images/interior-remodel/living_room_1.mask.png"), "Scandinavian"),
        (filepath("test-images/interior-remodel/living_room_2.png"), filepath("test-images/interior-remodel/living_room_2.mask.png"), "Rustic"),
    ]
    for seed in seed_list:
        for image, mask_image, preset in masked_testcases:
            suite.append(TestCase({
                "op": "interior-remodel",
                "image": image,
                "mask_image": mask_image,
                "preset": preset,
                "seed": seed,
            }))
    sets.append(suite)

    return sets


def paint_sets(seed_list):
    '''create the test suite'''
    sets = []

    suite = TestCaseSet(
        "paint-with-revision",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    simple_paint_testcases = [
        (filepath("test-images/paint-revise/wall1.png"), filepath("test-images/paint-revise/wall1.mask.png"), True, "interior"),
        (filepath("test-images/paint-revise/wall2.png"), filepath("test-images/paint-revise/wall2.mask.png"), True, "interior"),
        (filepath("test-images/paint-revise/wall3.png"), filepath("test-images/paint-revise/wall3.mask.png"), None, "interior"),
        (filepath("test-images/paint-revise/wall4.png"), filepath("test-images/paint-revise/wall4.mask.png"), None, "interior"),
        (filepath("test-images/paint-revise/wall1.png"), filepath("test-images/paint-revise/wall1.mask.png"), False, "interior"),
        (filepath("test-images/paint-revise/wall4.png"), filepath("test-images/paint-revise/wall4.mask.png"), False, "interior"),
        # (filepath("test-images/paint-revise/cabinet1.png"), filepath("test-images/paint-revise/cabinet1.mask.png"), None, "interior"),
        # (filepath("test-images/paint-revise/cabinet2.png"), filepath("test-images/paint-revise/cabinet2.mask.png"), None, "interior"),
        (filepath("test-images/paint-revise/cabinet2.png"), filepath("test-images/paint-revise/cabinet2.mask.png"), False, "interior"),
        (filepath("test-images/paint-revise/cabinet2.png"), filepath("test-images/paint-revise/cabinet2.mask.png"), True, "interior"),
        (filepath("test-images/paint-revise/cabinet3.png"), filepath("test-images/paint-revise/cabinet3.mask.png"), True, "interior"),
        (filepath("test-images/paint-revise/door1.png"), filepath("test-images/paint-revise/door1.mask.png"), True, "exterior"),
        (filepath("test-images/paint-revise/outsidewall1.png"), filepath("test-images/paint-revise/outsidewall1.mask.png"), True, "exterior"),
        (filepath("test-images/paint-revise/outsidewall2.png"), filepath("test-images/paint-revise/outsidewall2.mask.png"), True, "exterior"),
        (filepath("test-images/paint-revise/outsidewall3.png"), filepath("test-images/paint-revise/outsidewall3.mask.png"), True, "exterior"),
    ]
    for seed in seed_list:
        for image, mask_image, keep_surface_details, location in simple_paint_testcases:
            suite.append(TestCase({
                "op": "paint",
                "image": image,
                "mask_image": mask_image,
                "seed": seed,
                "keep_surface_details": keep_surface_details,
                "color_r": 112,
                "color_g": 107,
                "color_b": 83,
                "location": location,
            }))
    sets.append(suite)

    suite = TestCaseSet(
        "paint-exact",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    simple_paint_testcases = [
        (filepath("test-images/paint-no-revise/image_1.png"), filepath("test-images/paint-no-revise/image_1.mask.png"), True, "interior", 112, 107, 83),
        (filepath("test-images/paint-no-revise/image_1.png"), filepath("test-images/paint-no-revise/image_1.mask.png"), False, "interior", 112, 107, 83),
        (filepath("test-images/paint-no-revise/image_2.png"), filepath("test-images/paint-no-revise/image_2.mask.png"), True, "exterior", 150, 120, 90),
        (filepath("test-images/paint-no-revise/image_2.png"), filepath("test-images/paint-no-revise/image_2.mask.png"), False, "exterior", 150, 120, 90),
        (filepath("test-images/paint-no-revise/image_3.png"), filepath("test-images/paint-no-revise/image_3.mask.png"), True, "interior", 200, 180, 160),
        (filepath("test-images/paint-no-revise/image_3.png"), filepath("test-images/paint-no-revise/image_3.mask.png"), False, "interior", 200, 180, 160),
        (filepath("test-images/paint-no-revise/image_4.png"), filepath("test-images/paint-no-revise/image_4.mask.png"), True, "interior", 90, 100, 110),
        (filepath("test-images/paint-no-revise/image_4.png"), filepath("test-images/paint-no-revise/image_4.mask.png"), False, "interior", 90, 100, 110),
        (filepath("test-images/paint-no-revise/image_5.png"), filepath("test-images/paint-no-revise/image_5.mask.png"), True, "interior", 130, 140, 150),
        (filepath("test-images/paint-no-revise/image_5.png"), filepath("test-images/paint-no-revise/image_5.mask.png"), False, "interior", 130, 140, 150),
    ]
    for seed in seed_list:
        for image, mask_image, keep_surface_details, location, color_r, color_g, color_b in simple_paint_testcases:
            suite.append(TestCase({
                "op": "paint",
                "revise_mask": False,
                "image": image,
                "mask_image": mask_image,
                "seed": seed,
                "keep_surface_details": keep_surface_details,
                "location": location,
                "color_r": color_r,
                "color_g": color_g,
                "color_b": color_b,
            }))
    sets.append(suite)


    return sets


def replace_sets(seed_list):
    '''create the test suite'''
    sets = []

    suite = TestCaseSet(
        "replace",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )

    simple_replace_testcases = [
        (filepath("test-images/replace/stock_photo.png"), filepath("test-images/replace/stock_photo.mask.png"), "a light tan microsuede couch"),
        (filepath("test-images/replace/stock_photo.png"), filepath("test-images/replace/stock_photo.mask.png"), "a toy car"),
        (filepath("test-images/replace/christmas_tree.png"), filepath("test-images/replace/christmas_tree.mask.png"), "a christmas tree"),
        (filepath("test-images/replace/artwork.png"), filepath("test-images/replace/artwork.mask.png"), "modern artwork on the wall"),
        (filepath("test-images/replace/microwave.png"), filepath("test-images/replace/microwave.mask.png"), "microwave"),
        (filepath("test-images/replace/top_microwave.png"), filepath("test-images/replace/top_microwave.mask.png"), "a cabinet mounted microwave attached to the upper cabinet"),
        (filepath("test-images/replace/tall_cabinets.png"), filepath("test-images/replace/tall_cabinets.mask.png"), "tall cabinets that extend all the way to the ceiling"),
    ]

    for seed in seed_list:
        for image, mask_image, prompt in simple_replace_testcases:
            suite.append(TestCase({
                "op": "replace",
                "image": image,
                "mask_image": mask_image,
                "prompt": prompt,
                "seed": seed,
            }))
    sets.append(suite)

    return sets


def reskin_sets(seed_list):
    '''create the test suite'''
    sets = []

    suite = TestCaseSet(
        "reskin",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )

    simple_reskin_testcases = [
        (filepath("test-images/reskin/chair_1.png"), filepath("test-images/reskin/chair_1.mask.png"), "white couch"),
        (filepath("test-images/reskin/chair_1.png"), filepath("test-images/reskin/chair_1.mask.png"), "cowhide leather couch"),
        (filepath("test-images/reskin/chair_2.png"), filepath("test-images/reskin/chair_2.mask.png"), "black leather and wooden chair"),
        (filepath("test-images/reskin/bed.png"), filepath("test-images/reskin/bed.mask.png"), "red bed"),
        (filepath("test-images/reskin/couch.png"), filepath("test-images/reskin/couch.mask.png"), "yellow couch"),
        (filepath("test-images/reskin/couch.png"), filepath("test-images/reskin/couch.mask.png"), "violet furry couch"),
        (filepath("test-images/reskin/wall.png"), filepath("test-images/reskin/wall.mask.png"), "blue wall"),
        (filepath("test-images/reskin/wall.png"), filepath("test-images/reskin/wall.mask.png"), "red brick wall"),
        (filepath("test-images/reskin/wardrobe.png"), filepath("test-images/reskin/wardrobe.mask.png"), "birch wood wardrobe"),
    ]

    for seed in seed_list:
        for image, mask_image, prompt in simple_reskin_testcases:
            suite.append(TestCase({
                "op": "reskin",
                "image": image,
                "mask_image": mask_image,
                "prompt": prompt,
                "seed": seed,
            }))
    sets.append(suite)

    return sets


def remove_sets(seed_list):
    '''create the test suite'''
    sets = []

    suite = TestCaseSet(
        "remove",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )

    simple_remove_testcases = [
        (filepath("test-images/remove/bathroom.png"), filepath("test-images/remove/bathroom.mask.png")),
        (filepath("test-images/remove/bed.png"), filepath("test-images/remove/bed.mask.png")),
        (filepath("test-images/remove/bedroom1.png"), filepath("test-images/remove/bedroom1.mask.png")),
        (filepath("test-images/remove/bedroom2.png"), filepath("test-images/remove/bedroom2.mask.png")),
        (filepath("test-images/remove/cabinet.png"), filepath("test-images/remove/cabinet.mask.png")),
        (filepath("test-images/remove/couch.png"), filepath("test-images/remove/couch.mask.png")),
        (filepath("test-images/remove/countertop.png"), filepath("test-images/remove/countertop.mask.png")),
        (filepath("test-images/remove/fridge.png"), filepath("test-images/remove/fridge.mask.png")),
        (filepath("test-images/remove/kitchen.png"), filepath("test-images/remove/kitchen.mask.png")),
        (filepath("test-images/remove/tree.png"), filepath("test-images/remove/tree.mask.png")),
        (filepath("test-images/remove/wall1.png"), filepath("test-images/remove/wall1.mask.png")),
        (filepath("test-images/remove/wall2.png"), filepath("test-images/remove/wall2.mask.png")),
        (filepath("test-images/remove/wardrobe.png"), filepath("test-images/remove/wardrobe.mask.png"))
    ]

    for seed in seed_list:
        for image, mask_image in simple_remove_testcases:
            suite.append(TestCase({
                "op": "remove",
                "image": image,
                "mask_image": mask_image,
                "seed": seed,
            }))
    sets.append(suite)

    return sets


def floor_sets(seed_list):
    '''create the test suite'''
    sets = []

    suite = TestCaseSet(
        "floor-preset",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    testcases = [
        (filepath("../test-images/floor/floor1.png"),  "carpet"),
        (filepath("../test-images/floor/floor2.png"),  "concrete"),
        (filepath("../test-images/floor/floor3.png"),  "stained concrete"),
        (filepath("../test-images/floor/floor1.png"),  "tile"),
        (filepath("../test-images/floor/floor2.png"),  "travertine tile"),
        (filepath("../test-images/floor/floor3.png"),  "wood"),
        (filepath("../test-images/floor/floor1.png"),  "dark wood"),
        (filepath("../test-images/floor/floor2.png"),  "light wood"),
    ]
    for seed in seed_list:
        for image, preset in testcases:
            suite.append(TestCase({
                "op": "floor",
                "image": image,
                "preset": preset,
                "seed": seed,
            }))
    sets.append(suite)

    suite = TestCaseSet(
        "floor-prompt",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    testcases = [
        (filepath("../test-images/floor/floor1.png"),  "a room with beautiful large travertine tile"),
        (filepath("../test-images/floor/floor3.png"),  "a room with wool carpet and beautiful rug"),
    ]
    for seed in seed_list:
        for image, prompt in testcases:
            suite.append(TestCase({
                "op": "floor",
                "image": image,
                "prompt": prompt,
                "seed": seed,
            }))
    sets.append(suite)
    
    suite = TestCaseSet(
        "floor-style",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    testcases = [
        (filepath("../test-images/floor/floor1.png"), filepath("../test-images/floor/floor.style_1.png")),
        (filepath("../test-images/floor/floor1.png"), filepath("../test-images/floor/floor.style_2.png")),
        (filepath("../test-images/floor/floor2.png"), filepath("../test-images/floor/floor.style_1.png")),
        (filepath("../test-images/floor/floor2.png"), filepath("../test-images/floor/floor.style_2.png")),
        (filepath("../test-images/floor/floor3.png"), filepath("../test-images/floor/floor.style_1.png")),
        (filepath("../test-images/floor/floor3.png"), filepath("../test-images/floor/floor.style_2.png")),
    ]
    for seed in seed_list:
        for image, style_image in testcases:
            suite.append(TestCase({
                "op": "floor",
                "image": image,
                "style_image": style_image,
                "seed": seed,
            }))
    sets.append(suite)

    return sets


def wall_sets(seed_list):
    '''create the test suite'''
    sets = []

    suite = TestCaseSet(
        "wall-preset",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    testcases = [
        (filepath("../test-images/wall/wall1.png"), "board and batten"),
        (filepath("../test-images/wall/wall2.png"), "brick"),
        (filepath("../test-images/wall/wall3.png"), "painted"),
        (filepath("../test-images/wall/wall4.png"), "stone"),
        (filepath("../test-images/wall/wall1.png"), "wallpaper"),
        (filepath("../test-images/wall/wall2.png"), "wood cladding"),
        (filepath("../test-images/wall/wall3.png"), "wood paneling"),
        (filepath("../test-images/wall/wall4.png"), "wood slats"),
    ]
    for seed in seed_list:
        for image, preset in testcases:
            suite.append(TestCase({
                "op": "wall",
                "image": image,
                "preset": preset,
                "seed": seed,
            }))
    sets.append(suite)

    suite = TestCaseSet(
        "wall-prompt",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    testcases = [
        (filepath("../test-images/wall/wall3.png"), "a room with beautiful blue checkerboard pattern board and batten walls"),
        (filepath("../test-images/wall/wall2.png"), "a room with beautiful multi-color brick"),
    ]
    for seed in seed_list:
        for image, prompt in testcases:
            suite.append(TestCase({
                "op": "wall",
                "image": image,
                "prompt": prompt,
                "seed": seed,
            }))
    sets.append(suite)

    suite = TestCaseSet(
        "wall-style",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    testcases = [
        (filepath("../test-images/wall/wall1.png"), filepath("../test-images/wall/wall_style_1.png")),
        (filepath("../test-images/wall/wall1.png"), filepath("../test-images/wall/wall_style_2.png")),
        (filepath("../test-images/wall/wall2.png"), filepath("../test-images/wall/wall_style_1.png")),
        (filepath("../test-images/wall/wall2.png"), filepath("../test-images/wall/wall_style_2.png")),
        (filepath("../test-images/wall/wall3.png"), filepath("../test-images/wall/wall_style_1.png")),
        (filepath("../test-images/wall/wall3.png"), filepath("../test-images/wall/wall_style_2.png")),
    ]
    for seed in seed_list:
        for image, style_image in testcases:
            suite.append(TestCase({
                "op": "wall",
                "image": image,
                "style_image": style_image,
                "seed": seed,
            }))
    sets.append(suite)

    return sets


def classify_room_sets(seed_list):
    '''create the test suite'''
    sets = []

    suite = TestCaseSet(
        "classify-room",
        success_fn=lambda outputs : True,
    )

    # no seed input for classify room
    for image in TestCaseSet.create_file_list("test-images/classify-room/"):
        room_type = _get_room_type_from_filename(image)
        def check_room_type(outputs, room_type):
            print(f"Checking {outputs.get('room_type')} for {room_type}")
            return room_type in str(outputs.get('room_type'))

        testcase = TestCase(
            { "op": "classify-room", "image": image, },
            success_fn = partial(check_room_type, room_type=room_type),
        )
        suite.append(testcase)

    sets.append(suite)

    return sets


def put_it_here_sets(seed_list):
    '''create the test suite'''
    sets = []

    suite = TestCaseSet(
        "put-it-here",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )

    sample_inputs = [
        ('bg_1.png', 'table' , 'object_1.png', 512, 600,384,384 ),
        ('bg_2.png', 'table' , 'object_1.png', 350, 350,256,256 ),
        ('bg_3.png', 'table' , 'object_1.png', 350, 350,256,256 )
    ]
    for seed in seed_list:
        for filename, object_type, object_image_path, x_value, y_value, width_value, height_value in sample_inputs:
            suite.append(TestCase({
                "op": "put_it_here",
                "object_type": object_type,
                "seed": seed,
                "image": filepath(f"test-images/staging/put-it-here/{filename}"),
                "object_image":  filepath(f"test-images/staging/put-it-here/{object_image_path}"),
                "x_value" : x_value,
                "y_value" : y_value,
                "width_value" : width_value,
                "height_value" : height_value,
            }))
    sets.append(suite)

    return sets


def multiview_sets(seed_list):
    '''create the test suite'''
    sets = []

    suite = TestCaseSet(
        "multiview",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )

    # no seed
    for image in TestCaseSet.create_file_list("test-images/staging/multiview/"):
        suite.append(TestCase({
            "op": "multiview",
            "image": image,
        }))
    sets.append(suite)

    return sets


def roof_sets(seed_list):
    '''create the test suite'''
    sets = []

    suite = TestCaseSet(
        "roof",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )

    roof_configs = [
        ("brava", "aged_cedar_shake"),
        ("owens_corning", "aged_copper"),
        ("owens_corning", "black_sable"),
        ("owens_corning", "bourbon"),
        ("owens_corning", "brownwood"),
        ("owens_corning", "carbon"),
        ("owens_corning", "chateau_green"),
        ("owens_corning", "desert_tan"),
        ("owens_corning", "driftwood"),
        ("owens_corning", "estate_gray"),
        ("owens_corning", "forest_brown"),
        ("owens_corning", "granite"),
        ("owens_corning", "harbor_blue"),
        ("owens_corning", "merlot"),
        ("owens_corning", "mesquite"),
        ("owens_corning", "midnight_plum"),
        ("owens_corning", "mountainside"),
        ("owens_corning", "night_sky"),
        ("owens_corning", "onyx_black"),
        ("owens_corning", "pacific_wave"),
        ("owens_corning", "sand_castle"),
        ("owens_corning", "sand_dune"),
        ("owens_corning", "sedona_canyon"),
        ("owens_corning", "smokey_mountain"),
        ("owens_corning", "storm_cloud"),
        ("owens_corning", "summer_harvest"),
        ("owens_corning", "summerwood"),
        ("owens_corning", "sycamore"),
        ("owens_corning", "teak"),
        ("owens_corning", "terra_cotta"),
        ("owens_corning", "williamsburg_gray")
    ]

    for seed in seed_list:
        for image in TestCaseSet.create_file_list("test-images/roof/"):
            for roof_type, roof_name in roof_configs:
                for light_condition in ["full_sun", "cloudy"]:
                    suite.append(TestCase({
                        "op": "roof",
                        "roof_type": roof_type,
                        "roof_name": roof_name,
                        "light_condition": light_condition,
                        "seed": seed,
                        "image": image,
                    }))
    sets.append(suite)

    return sets


def example_sets(seed_list):
    '''create the test suite'''
    sets = []

    suite = TestCaseSet(
        "example",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )

    for seed in seed_list:
        for i in [1,2]:
            suite.append(TestCase({
                "op": "example",
                "seed": seed,
                "foo": i,
            }))
    sets.append(suite)

    return sets


def create_sets(seed_list):
    """create all test sets"""
    sets = [
        *staging_sets(seed_list),
        *unstaging_sets(seed_list),
        *interior_remodel_sets(seed_list),
        *paint_sets(seed_list),
        *replace_sets(seed_list),
        *reskin_sets(seed_list),
        *remove_sets(seed_list),
        *floor_sets(seed_list),
        *wall_sets(seed_list),
        *classify_room_sets(seed_list),
        *put_it_here_sets(seed_list),
        *multiview_sets(seed_list),
        *roof_sets(seed_list),
    ]
    return sets
