"""reimage service helper functions"""
import os
import io
import sys
import time
import datetime
import json
import tempfile
import traceback
import uuid
import urllib
import shutil
import zipfile
import socket
from pathlib import Path
import requests
from dateutil.tz import tzoffset

TIMEOUT = 5
VERIFY=True
SAVE_OUTPUTS = False
SAVE_INPUTS = False
SEND_EXTRA = False

class Job:
    """stores all the information about a job"""
    def __init__(self, host):
        self.job_id = 0
        self.inputs = {}
        self.params = {}
        self.json = {}
        self.host = host
        self.base_url = f"https://{host}"

        # we store the session in the job, so that all subsequent API calls
        # use the same session for performance reasons
        # this also prevents any issue with connecting to another broker should the DNS change
        # we can't have a global session in case the DNS does change, so one per job is ideal
        self.session = requests.Session()

    def __str__(self):
        return f"Job[{self.job_id}]"

    def close(self):
        if self.session is not None:
            self.session.close()
        for _, v in self.inputs.items():
            if isinstance(v,str) and os.path.exists(v):
                os.remove(v)

# the active_job (if any)
active_job = None

# the time we started waiting for a job (idle time)
last_job_time = time.time()

# a string to represent the identity of this worker
ident = socket.gethostname()


def tmpname(prefix="tmpfile", suffix=".tmp"):
    """generate a good tmpfile name"""
    random=uuid.uuid4().hex[:8]
    return f"/tmp/{prefix}{random}{suffix}"


def get_job(host, job_name, priority, timeout_sec, hints=None):
    """get a job"""
    try:
        job = Job(host)
        hints_arg = f"&hints={','.join(hints)}" if hints is not None else ""
        resp = job.session.get(f"{job.base_url}/job/{job_name}/waitzip?priority={priority}&timeout={round(timeout_sec*1000)}&ident={ident}{hints_arg}", verify=VERIFY, timeout=30)
        if resp.status_code not in [200, 204]:
            print(f"get_job: Invalid response from broker: {resp.status_code}", flush=True)
            time.sleep(timeout_sec)
            job.close()
            return None
        if resp.headers.get('content-type') == 'application/json':
            print(f"get_job: Response from broker: {resp.content}")
            time.sleep(timeout_sec)
            job.close()
            return None
        if resp.content == b'':
            # no job available
            job.close()
            return None

        files = {}
        with zipfile.ZipFile(io.BytesIO(resp.content)) as respzip:
            for name in respzip.namelist():
                if name == "job.json":
                    job.json = json.loads(respzip.read(name))
                else:
                    filedesc, path = tempfile.mkstemp(suffix=".png",prefix=f"input-{name}-")
                    with os.fdopen(filedesc, 'wb') as tmpo:
                        tmpo.write(respzip.read(name))
                    files[name] = path

        if job.json is None:
            print("Invalid job: No JSON", flush=True)
            time.sleep(timeout_sec)
            job.close()
            return None
        if job.json.get('jobID') is None:
            print("Invalid job: no job ID", flush=True)
            time.sleep(timeout_sec)
            job.close()
            return None
        job.job_id = job.json.get('jobID')
        job.inputs = job.json.get('inputs') if job.json.get('inputs') is not None else {}
        for k,val in files.items():
            job.inputs[k] = Path(val)

        # params are deprecated - this can be removed
        job.params = job.json.get('params') if job.json.get('params') is not None else {}

        return job
    except zipfile.BadZipFile:
        traceback.print_exc(file=sys.stdout)
        print(f"Response: {len(resp.content)} {str(resp.content[:50])}...")
        time.sleep(timeout_sec)
        job.close()
        raise
    except:
        traceback.print_exc(file=sys.stdout)
        time.sleep(timeout_sec)
        job.close()
        raise


def update_job(job, status, outputs=None, detail=None, percent=None, warning=None):
    """update job"""
    if job is None or job.job_id == 0:
        return 0
    js = {}
    js['jobID'] = job.job_id
    js['status'] = status
    if percent is not None:
        js['percent'] = percent
    if isinstance(outputs, dict) and outputs.get('detail') is not None:
        detail = outputs.get('detail')
    if detail is not None:
        js['detail'] = detail
    if isinstance(outputs, dict) and outputs.get('warning') is not None:
        warning = outputs.get('warning')
    if warning is not None:
        js['warning'] = warning
    file_outputs = {}
    try:
        time1 = time.time()
        file_outputs = {}
        if isinstance(outputs, dict):
            js['outputs'] = {}
            # FIXME historically we used 'detail' to return non-file results
            # unfortunately the app now looks for results in js['detail']
            # In an effort to support any type of output - lets start by putting the detail in the ouputs
            # Newer version of the app will check js['outputs']['detail'] and then over time we can move to just using a properly named output
            if detail is not None:
                js['outputs']['detail'] = detail
            if warning is not None:
                js['outputs']['warning'] = warning
            for key, value in outputs.items():
                # if a file - add it to the multipart data
                if isinstance(value, str) and os.path.exists(value):
                    if os.stat(value).st_size == 0:
                        continue # skip if empty
                    if key.startswith('extra') and not SEND_EXTRA:
                        continue
                    # pylint: disable=consider-using-with
                    filedesc = open(value, 'rb')
                    file_outputs[key] = filedesc
                    if SAVE_OUTPUTS:
                        path = tmpname(suffix=".png",prefix=f"{job.job_id}-{key}-")
                        shutil.copy(value, path)
                # if not a file, just put the value in the outputs dict
                else:
                    if key.startswith('extra') and not SEND_EXTRA:
                        continue
                    js['outputs'][key] = value

        # build multipart data
        multipart_form_data = {
            'json': (None, json.dumps(js)),
        }
        for key, value in file_outputs.items():
            multipart_form_data[key] = (key, value)

        r = job.session.post(f"{job.base_url}/job/{job.job_id}/update?ident={ident}", files=multipart_form_data, verify=VERIFY, timeout=30)
        if r.status_code != 200:
            print(f"update_job: Invalid response from broker: {r.status_code}", flush=True)
        time2 = time.time()
        print(f"Update Job {job.job_id} {status} ({time2-time1:0.2f}s)", flush=True)
        rjson = r.json()
        if rjson.get('elapsedTime') is not None:
            return rjson.get('elapsedTime')
        return 0
    except:
        traceback.print_exc(file=sys.stdout)
    finally:
        for _, filedesc in file_outputs.items():
            filedesc.close()
    return 0


def update_job_error(job, error_str):
    """update job with the specified error string"""
    print(error_str, flush=True)
    update_job(job, "error", detail=f"error: {error_str}")


def requeue_active_job():
    """requeue a job for another worker to handle"""
    if active_job is None:
        return
    try:
        resp = active_job.session.post(f"{active_job.base_url}/job/{active_job.job_id}/requeue?ident={ident}", verify=VERIFY, timeout=30)
        if resp.status_code not in [200, 204]:
            print(f"requeue: Invalid response from broker: {resp.status_code}", flush=True)
        else:
            print(f"Requeued job {active_job.job_id}")
        return
    except:
        traceback.print_exc(file=sys.stdout)
        return


def notify(host, message):
    """just sends a message for logging"""
    try:
        message = urllib.parse.quote(message)
        resp = requests.post(f"https://{host}/notify?ident={ident}&message={message}", verify=VERIFY, timeout=30)
        if resp.status_code not in [200, 204]:
            print(f"notify: Invalid response from broker: {resp.status_code}", flush=True)
        return
    except:
        traceback.print_exc(file=sys.stdout)
        return


def dojob(host, job_name, process, priority, timeout, hints=None):
    """get and execute a job"""
    # pylint: disable=global-statement
    global last_job_time
    global active_job

    time1 = time.time()
    job = get_job(host, job_name, priority, timeout, hints=hints)
    time2 = time.time()
    if job is None:
        return

    print(f"=========  {datetime.datetime.now(tzoffset(None, -8.0 * 3600)).strftime('%Y-%m-%dT%H:%M:%S.%f')} (fetch: {time2-time1:0.2f}s) (idle: {time.time()-last_job_time:0.2f}s) =========")
    active_job = job

    try:
        time1 = time.time()
        # params are deprecated
        process(job, **job.inputs, **job.params)
        time2 = time.time()
        print(f"=========  {datetime.datetime.now(tzoffset(None, -8.0 * 3600)).strftime('%Y-%m-%dT%H:%M:%S.%f')} (total process time: {time2-time1:0.2f}s)  =========\n")

    except Exception as exc:
        traceback.print_exc(file=sys.stdout)
        update_job(job, 'error')
        raise exc
    finally:
        last_job_time = time.time()
        if SAVE_INPUTS:
            for key, filename in job.inputs.items():
                if os.path.exists(filename):
                    path = tmpname(suffix=".png",prefix=f"{job.job_id}-input-{key}-")
                    shutil.copy(filename, path)
        active_job = None
        job.close()
