"""exterior remodel functions"""
import re
import os
import tempfile

import reimage
import subproc
import subproc_util
import pipelines
from util import check_image_sizes
from operations import (
    canny,
    depth,
    t2i_adapter,
    inpainting,
    controlnet,
)

def remodel_exterior(
        job_id,
        batch_size,
        image,
        mask_image,
        style_image,
        preset,
        prompt,
        negative_prompt,
        structure_type,
        seed,
        fidelity,
):
    if job_id is None:
        raise Exception("job_id is required")
    if image is None:
        raise Exception("image is required")
    if prompt is None and style_image is None and preset is None:
        raise Exception("prompt, style_image, or preset required")
    if fidelity is None:
        fidelity = 0.65
    else:
        fidelity = float(fidelity)
    if negative_prompt is None:
        negative_prompt = "anime, cartoon, graphic, text, crayon, glitch, blur, ugly, deformed, blurry, watermark, disfigured"
    if structure_type is None:
        structure_type = "house"

    (width, height) = check_image_sizes(image=image, mask_image=mask_image, style_image=style_image)

    if prompt is None and preset is not None:
        # Craft a prompt from the preset
        if preset is not None:
            preset = re.sub('(.)([A-Z][a-z]+)', r'\1 \2', preset).lower()
        else:
            preset = ""
        if preset == "post modern":
            preset = "postmodern"
        prompt = f"A photograph of a beautiful {preset} {structure_type}, outside"
    # If we're copying the preset from an image just use a simple prompt
    if style_image is not None:
        prompt = f"{structure_type}, outside"
        # prompt = "beautiful"

    # if a mask is specified we need to inpaint, which is totally different
    # so called a specific function to handle that
    if mask_image is not None:
        result = remodel_exterior_masked(job_id, batch_size, image, mask_image, style_image, prompt, negative_prompt, seed, fidelity)
    else:
        result = remodel_exterior_unmasked(job_id, batch_size, image, style_image, prompt, negative_prompt, seed, fidelity, width, height)
    return result


def remodel_exterior_unmasked(
        job_id,
        batch_size,
        image,
        style_image,
        prompt,
        negative_prompt,
        seed,
        fidelity,
        width,
        height,
):
    depth_image_path = depth(image)
    canny_image_path = canny(image)

    lora_names=[]
    lora_weights=[]
    if " oc_black_sable " in prompt:
        lora_names.append("lora-black_sable-sdxl")
        lora_weights.append(1.0)
    # if " dark " in prompt:
    #     lora_names.append("lora-noise_offset")
    #     lora_weights.append(1.0)

    if style_image is None:
        pipeline = "sdxl_t2i_adapter"
        args = {
            "prompt": prompt,
            "adapter_types": ["depth", "canny"],
            "adapter_images_conditioned": [depth_image_path, canny_image_path],
            "adapter_scales": [min(fidelity,1.0)**0.7, fidelity**(1/3)],
            "batch_size": batch_size,
            "steps": 25,
            "lora_names": lora_names,
            "lora_weights": lora_weights,
            "guidance_scale": 7.5,
            "seed": seed,
            "negative_prompt": negative_prompt,
            "width": width,
            "height": height,
            "scheduler": "dpms",
        }
        subproc_util.subproc_launch(pipeline, job_id, fn=pipelines.pipeline_get_loader(pipeline))
        reimage.update_job(job_id, 'rendering')
        result = subproc.subproc_call(pipeline, t2i_adapter, args=args)
    else:
        pipeline = "sdxl_controlnet_ip_adapter"
        args = {
            "prompt": prompt,
            "control_types": ["depth-sdxl"],
            "control_images_conditioned": [depth_image_path],
            "control_scales": [fidelity],
            "ip_adapter_image": style_image,
            "ip_adapter_scale": fidelity,
            "batch_size": batch_size,
            "steps": 25,
            "guidance_scale": 7.5,
            "seed": seed,
            "negative_prompt": negative_prompt,
            "width": width,
            "height": height,
            "scheduler": "dpms",
        }
        subproc_util.subproc_launch(pipeline, job_id, fn=pipelines.pipeline_get_loader(pipeline))
        reimage.update_job(job_id, 'rendering')
        result = subproc.subproc_call(pipeline, controlnet, args=args)

    if isinstance(result, BaseException):
        raise result
    if isinstance(result, dict):
        return {**result, "extra-0-orig": image, "extra-1-style": style_image, "extra-2-canny": canny_image_path, "extra-3-depth": depth_image_path}
    return result


def remodel_exterior_masked(
        job_id,
        batch_size,
        image,
        mask_image,
        style_image,
        prompt,
        negative_prompt,
        seed,
        fidelity,
):
    depth_image_path = depth(image)
    canny_image_path = canny(image)

    if style_image is None:
        pipeline = "sdxl_inpainting_controlnet"
        args = {
            "prompt": prompt,
            "control_types": ["depth-sdxl", "canny-sdxl"],
            "control_scales": [fidelity, fidelity],
            "control_images_conditioned": [depth_image_path, canny_image_path],
            "batch_size": batch_size,
            "steps": 25,
            "guidance_scale": 10.0,
            "seed": seed,
            "negative_prompt": negative_prompt,
            "scheduler": "dpms",
            "input_image": image,
            "mask_image": mask_image,
            "overmask": True,
        }
        subproc_util.subproc_launch(pipeline, job_id, fn=pipelines.pipeline_get_loader(pipeline))
        reimage.update_job(job_id, 'rendering')
        result = subproc.subproc_call(pipeline, inpainting, args=args)
    else:
        pipeline = "sdxl_inpainting_controlnet_ip_adapter"
        args = {
            "prompt": prompt,
            "control_types": ["depth-sdxl"],
            "control_scales": [fidelity],
            "control_images_conditioned": [depth_image_path],
            "ip_adapter_image": style_image,
            "ip_adapter_scale": fidelity,
            "batch_size": batch_size,
            "steps": 25,
            "guidance_scale": 10.0,
            "strength": 0.999,
            "seed": seed,
            "negative_prompt": negative_prompt,
            "scheduler": "dpms",
            "input_image": image,
            "mask_image": mask_image,
            "overmask": True,
        }
        subproc_util.subproc_launch(pipeline, job_id, fn=pipelines.pipeline_get_loader(pipeline))
        reimage.update_job(job_id, 'rendering')
        result = subproc.subproc_call(pipeline, inpainting, args=args)

    if isinstance(result, BaseException):
        raise result
    if isinstance(result, dict):
        return {**result, "extra-0-orig": image, "extra-1-style": style_image, "extra-2-mask": mask_image, "extra-3-canny": canny_image_path, "extra-4-depth": depth_image_path}
    return result
