"""reimage service helper functions"""
import os
import io
import sys
import time
import datetime
import json
import tempfile
import traceback
import requests
import urllib
import shutil
import zipfile
import socket
from dateutil.tz import tzoffset

BASE_URL = ""
TIMEOUT = 5
VERIFY=True
SAVE_RESULTS = False
SAVE_INPUTS = False
SEND_EXTRA = False

session = requests.Session()
active_job_id = None

# the time we started waiting for a job (idle time)
last_job_time = time.time()

# a string to represent the identity of this worker
ident = socket.gethostname()

def get_job(job_name, priority, timeout_sec, hints=None):
    """get a job"""
    try:
        hints_arg = f"&hints={','.join(hints)}" if hints is not None else ""
        t1 = time.time()
        resp = session.get(f"{BASE_URL}/job/{job_name}/waitzip?priority={priority}&timeout={round(timeout_sec*1000)}&ident={ident}{hints_arg}", verify=VERIFY, timeout=30)
        t2 = time.time()
        if resp.status_code not in [200, 204]:
            print(f"get_job: Invalid response from broker: {resp.status_code}", flush=True)
            time.sleep(timeout_sec)
            return None, None, None
        if resp.headers.get('content-type') == 'application/json':
            print(f"get_job: Response from broker: {resp.content}")
            return None, None, None
        if resp.content == b'':
            return None, None, None

        jobjson = None
        inputs = {}

        with zipfile.ZipFile(io.BytesIO(resp.content)) as respzip:
            for name in respzip.namelist():
                if name == "job.json":
                    jobjson = json.loads(respzip.read(name))
                else:
                    fd, path = tempfile.mkstemp(suffix=".png",prefix=f"input-{name}-")
                    with os.fdopen(fd, 'wb') as tmpo:
                        tmpo.write(respzip.read(name))
                    inputs[name] = path
        return jobjson, inputs, t2-t1
    except zipfile.BadZipFile:
        traceback.print_exc(file=sys.stdout)
        print(f"Response: {len(resp.content)} {str(resp.content[:50])}...")
        time.sleep(timeout_sec)
        return None, None, None
    except:
        traceback.print_exc(file=sys.stdout)
        time.sleep(timeout_sec)
        return None, None, None


def update_job(job_id, status, results=None, detail=None, percent=None, warning=None):
    """update job"""
    if job_id == 0:
        return
    js = {}
    js['jobID'] = job_id
    js['status'] = status
    if detail is not None:
        js['detail'] = detail
    if percent is not None:
        js['percent'] = percent
    if warning is not None:
        js['warning'] = warning
    open_files = []
    try:
        t1 = time.time()
        multipart_form_data = {
            'json': (None, json.dumps(js)),
        }
        if results != {} and results is not None:
            for key, filename in results.items():
                if not isinstance(filename, str):
                    continue
                if os.stat(filename).st_size == 0:
                    continue # skip if empty
                if key.startswith('extra') and not SEND_EXTRA:
                    continue
                f = open(filename, 'rb')
                open_files.append(f)
                multipart_form_data[key] = (key, f)
                if SAVE_RESULTS:
                    _, path = tempfile.mkstemp(suffix=".png",prefix=f"{job_id}-{key}-")
                    shutil.copy(filename, path)

        r = session.post(f"{BASE_URL}/job/{job_id}/update?ident={ident}", files=multipart_form_data, verify=VERIFY, timeout=30)
        t2 = time.time()
        print(f"Update Job {job_id} {status} ({t2-t1:0.2f}s)", flush=True)

        if status == "error":
            print(f"update_job: error - {detail}")
        if r.status_code != 200:
            print(f"update_job: Invalid response from broker: {r.status_code}", flush=True)
    except:
        traceback.print_exc(file=sys.stdout)
    finally:
        for f in open_files:
            f.close()


def update_job_error(job_id, error_str):
    """update job with the specified error string"""
    print(error_str, flush=True)
    update_job(job_id, "error", detail=f"error: {error_str}")


def requeue_active_job():
    """requeue a job for another worker to handle"""
    global active_job_id
    if active_job_id is None:
        return
    try:
        resp = session.post(f"{BASE_URL}/job/{active_job_id}/requeue?ident={ident}", verify=VERIFY, timeout=30)
        if resp.status_code not in [200, 204]:
            print(f"requeue: Invalid response from broker: {resp.status_code}", flush=True)
        else:
            print(f"Requeued job {active_job_id}")
        return
    except:
        traceback.print_exc(file=sys.stdout)
        return


def notify(message):
    """just sends a message for logging"""
    try:
        message = urllib.parse.quote(message)
        resp = session.post(f"{BASE_URL}/notify?ident={ident}&message={message}", verify=VERIFY, timeout=30)
        if resp.status_code not in [200, 204]:
            print(f"notify: Invalid response from broker: {resp.status_code}", flush=True)
        return
    except:
        traceback.print_exc(file=sys.stdout)
        return


def dojob(job_name, process, priority, timeout, hints=None):
    global active_job_id

    t1 = time.time()
    job_json, inputs, requesttime = get_job(job_name, priority, timeout, hints=hints)
    t2 = time.time()
    if job_json is None:
        return
    if job_json.get('jobID') is None:
        print("Invalid job: jobID", flush=True)
        return
    job_id = job_json.get('jobID')

    global last_job_time
    print(f"=========  {datetime.datetime.now(tzoffset(None, -8.0 * 3600)).strftime('%Y-%m-%dT%H:%M:%S.%f')} (request: {requesttime:0.2f}s) (request+parse: {t2-t1:0.2f}s) (idle: {time.time()-last_job_time:0.2f}s) =========")
    active_job_id = job_id
    params = job_json.get('params') if job_json.get('params') is not None else {}

    try:
        t1 = time.time()
        process(job_id, **inputs, **params)
        t2 = time.time()
        print(f"=========  {datetime.datetime.now(tzoffset(None, -8.0 * 3600)).strftime('%Y-%m-%dT%H:%M:%S.%f')} (total process time: {t2-t1:0.2f}s)  =========\n")

    except Exception as e:
        traceback.print_exc(file=sys.stdout)
        update_job(job_id, 'error')
        raise e
    finally:
        last_job_time = time.time()
        active_job_id = None
        if SAVE_INPUTS:
            for key, filename in inputs.items():
                _, path = tempfile.mkstemp(suffix=".png",prefix=f"{job_id}-input-{key}-")
                shutil.copy(filename, path)
        for _, v in inputs.items():
            if os.path.exists(v):
                os.remove(v)
