"""subproc utilities specific for diffuser-models"""
import torch
import os
import sys
import time
import setproctitle
import signal
import traceback

import subproc
import reimage

def subproc_setup(title):
    """setup the subprocess and load the pipelines"""
    try:
        # set a new process title for new process
        setproctitle.setproctitle(title)

        # signal handlers
        signal.signal(signal.SIGINT, signal.SIG_IGN)

        # set offline mode
        os.environ["HF_HUB_OFFLINE"] = "1"

        # disable grad
        torch.set_grad_enabled(False)

        return
    except Exception as e:
        traceback.print_exc(file=sys.stdout)
        raise e


def subproc_fn_wrapper(subproc_name, fn, fn_args):
    print(f"Loading {subproc_name} ...")
    t1 = time.time()
    if fn_args is None:
        fn()
    else:
        fn(**fn_args)
    t2 = time.time()
    print(f"Loaded  {subproc_name} ({t2-t1:0.2f}s)", flush=True)


def subproc_launch(subproc_name, job_id, fn=None, fn_args=None):
    """launch a new subproc if needed, specify a optional function to run in the new subproc"""
    if not subproc.subproc_running(subproc_name):
        if job_id is not None:
            reimage.update_job(job_id, "loading model")
        result = subproc.subproc_start(subproc_name)
        if not result:
            raise Exception(f"Failed to start process: {subproc_name}")
        result = subproc.subproc_call(
            subproc_name,
            subproc_setup,
            args={"title": subproc_name},
        )
        if isinstance(result, BaseException):
            raise Exception(f"failed to setup subprocess: {subproc_name} {result}")

        if fn is not None:
            result = subproc.subproc_call(
                subproc_name,
                subproc_fn_wrapper,
                args={"subproc_name": subproc_name, "fn": fn, "fn_args": fn_args},
            )
            if isinstance(result, BaseException):
                raise Exception(f"failed to launch subprocess: {subproc_name} {result}")
