"""upscale functions"""
import tempfile
import cv2

import reimage
import subproc
import subproc_util
import pipelines
from PIL import Image
from operations import (
    img2img,
)

def upscale_controlnet_tile(job_id, image, prompt, seed, size, strength):
    if image is None:
        raise Exception("image is required")
    if size is None:
        raise Exception("size is required")
    if prompt is None:
        prompt = ""
    if strength is None:
        strength = 1.0
    if size > 2048:
        raise Exception(f"Upscale too large: {size}")
    height, width, _ = cv2.imread(image).shape
    if height * width > 2048 * 2048:
        raise Exception(f"image too large: {width}x{height}")

    image = Image.open(image).convert("RGB")
    condition_image = resize_image_for_upscale(image, size)
    _, condition_image_path = tempfile.mkstemp(
        suffix=".png", prefix="upscale-tile-condition-"
    )
    condition_image.save(condition_image_path)

    pipeline = "stable_diffusion_img2img_controlnet"
    args = {
        "input_image": condition_image_path,
        "prompt": prompt,
        "negative_prompt": "blur, lowres, text, worst quality, low quality, jpeg artifacts, ugly, deformed, blurry, watermark",
        "batch_size": 1,
        "steps": 25,
        "strength": strength,
        "guidance_scale": 5.0,
        # Since this commit
        # https://github.com/huggingface/diffusers/commit/8263cf00f832399bca215e29fa7572e0b0bde4da
        # unipc and dpms produce black images
        "scheduler": "euler-a",
        "seed": seed,
        "control_types": ["tile"],
        "control_images_conditioned": [condition_image_path],
    }
    subproc_util.subproc_launch(pipeline, job_id, fn=pipelines.pipeline_get_loader(pipeline))
    reimage.update_job(job_id, "rendering")
    result = subproc.subproc_call(pipeline, img2img, args=args)
    if isinstance(result, BaseException):
        raise result
    if isinstance(result, dict):
        return {
            **result,
            "extra-0-orig": image,
            "extra-1-resized": condition_image_path,
        }
    return result

def resize_image_for_upscale(input_image: Image, resolution: int):
    """Resize an image to the specified resolution, round to SD compatible size"""
    input_image = input_image.convert("RGB")
    width, height = input_image.size
    scale = float(resolution) / max(height, width)
    height *= scale
    width *= scale
    height = int(round(height / 64.0)) * 64
    width = int(round(width / 64.0)) * 64
    img = input_image.resize((width, height), resample=Image.LANCZOS)
    return img

