"""wall functions"""
import logging
logging.basicConfig(level=logging.ERROR)
logger = logging.getLogger(__name__)
import re

import reimage
import subproc
import subproc_util
import pipelines
from util import check_image_sizes
from operations import (
    depth,
    seg,
    inpainting,
)

def remodel_wall(
    job_id,
    batch_size,
    image,
    style_image,
    preset,
    prompt,
    negative_prompt,
    seed,
    fidelity,
):
    if job_id is None:
        raise Exception("job_id is required")
    if image is None:
        raise Exception("image is required")
    if prompt is None and style_image is None and preset is None:
        raise Exception("prompt, style_image, or preset required")

    if fidelity is None:
        fidelity = 0.65
    else:
        fidelity = float(fidelity)
    if negative_prompt is None:
        negative_prompt = "anime, cartoon, graphic, text, crayon, glitch, blur, ugly, deformed, blurry, watermark, disfigured"

    check_image_sizes(image=image)

    if prompt is None and preset is not None:
        # Craft a prompt from the preset
        preset = re.sub('(.)([A-Z][a-z]+)', r'\1 \2', preset).lower()
        prompt = f"A room with a {preset} walls"

    # Create the wall mask
    mask_image = seg(image=image, preset="wall", expand_pixels=30)

    depth_image_path = depth(image)

    if style_image is None:
        pipeline = "sdxl_inpainting_controlnet"
        args = {
            "prompt": prompt,
            "input_image": image,
            "mask_image": mask_image,
            "batch_size": batch_size,
            "steps": 25,
            "guidance_scale": 7.5,
            "seed": seed,
            "negative_prompt": negative_prompt,
            "overmask": True,
            "control_types": ["depth-sdxl"],
            "control_scales": [fidelity],
            "control_images_conditioned": [depth_image_path],
        }
    else:
        pipeline = "sdxl_inpainting_controlnet_ip_adapter"
        args = {
            "prompt": prompt,
            "input_image": image,
            "mask_image": mask_image,
            "ip_adapter_image": style_image,
            "ip_adapter_scale": fidelity,
            "batch_size": batch_size,
            "steps": 25,
            "guidance_scale": 10.0,
            "strength": 0.999,
            "seed": seed,
            "negative_prompt": negative_prompt,
            "scheduler": "dpms",
            "overmask": True,
            "control_types": ["depth-sdxl"],
            "control_scales": [fidelity],
            "control_images_conditioned": [depth_image_path],
        }
    subproc_util.subproc_launch(pipeline, job_id, fn=pipelines.pipeline_get_loader(pipeline))
    reimage.update_job(job_id, 'rendering')
    result = subproc.subproc_call(pipeline, inpainting, args=args)
    if isinstance(result, BaseException):
        raise result
    return result
