"""interior remodel functions"""
import re
import os
import random
import tempfile
import cv2
import numpy as np

import reimage
import subproc
import subproc_util
import pipelines
from util import check_image_sizes, auto_resize
from operations import (
    canny,
    depth,
    seg,
    t2i_adapter,
    inpainting,
    controlnet,
)

def remodel_interior(
        job,
        image,
        batch_size=4,
        preset=None,
        prompt=None,
        room_type=None,
        fidelity=None,
        mask_image=None,
        style_image=None,
        negative_prompt=None,
        seed=None,
):
    if image is None:
        raise Exception("image is required")
    if prompt is None and style_image is None and preset is None:
        raise Exception("prompt, style_image, or preset required")
    if fidelity is None:
        fidelity = 0.65
    else:
        fidelity = float(fidelity)
    if negative_prompt is None:
        negative_prompt = "anime, cartoon, graphic, text, crayon, glitch, blur, ugly, deformed, blurry, watermark, disfigured"

    (width, height) = check_image_sizes(image=image, mask_image=mask_image, style_image=style_image)

    # If not a good size - resize
    if width % 64 != 0 or height % 64 != 0 or width*height > 1024*1024*1.5:
        print(f"Resizing image: {width}x{height} is not ideal")
        image = auto_resize(image)
        (width, height) = check_image_sizes(image=image, mask_image=mask_image, style_image=style_image)
        print(f"Resized image: {width}x{height}")

    if prompt is None and preset is not None:
        if room_type is None:
            raise Exception("room_type is required for preset")
        # Craft a prompt from the preset (and room_type/room_type) if not supplied
        if preset is not None:
            preset = re.sub('(.)([A-Z][a-z]+)', r'\1 \2', preset).lower()
        else:
            preset = ""
        # in some cases we actually want to make adjustments for certain presets
        # For example art-deco makes images that look like they are from the 1920s
        # but we actually want "contemporary art deco"
        if preset == "post modern":
            preset = "postmodern"
        if preset in ["industrial"]:
            preset = f"contemporary {preset}--"
        if preset in ["mediterranean"]:
            preset = f"updated {preset}"
        if preset in ["art deco", "cottage", "french country", "rustic", "shabby chic", "southwestern", "tuscan", "bohemian"]:
            preset = f"contemporary {preset}"
        room_type = re.sub('(.)([A-Z][a-z]+)', r'\1 \2', room_type).lower()
        prompts = [
            f"a beautiful {preset} {room_type}, gorgeous floor, photorealistic",
            f"a beautiful {preset} {room_type}, gorgeous floor, photorealistic, high budget, designer",
            f"a beautiful {preset} {room_type}, gorgeous floor, (restoration hardware)",
            f"a beautiful {preset} {room_type}, gorgeous floor, pinterest, houzz",
            f"a beautiful {preset} {room_type}, gorgeous floor, (architecture digest), magazine",
        ]
        prompt = random.choice(prompts)
    # If we're copying the preset from an image just use a simple prompt
    if style_image is not None:
        prompt = f"{room_type}"
        # prompt = "beautiful"

    if mask_image is not None:
        # if a mask is specified we need to inpaint, which is totally different
        # so called a specific function to handle that
        result = remodel_interior_masked(job, batch_size, image, mask_image, style_image, prompt, negative_prompt, seed, fidelity, width, height)
    else:
        result = remodel_interior_unmasked(job, batch_size, image, style_image, prompt, negative_prompt, seed, fidelity, width, height)
    return result


def remodel_interior_unmasked(
        job,
        batch_size,
        image,
        style_image,
        prompt,
        negative_prompt,
        seed,
        fidelity,
        width,
        height,
):
    depth_image_path = depth(image)
    floor_image_path = seg(image, preset="floor", expand=-1)
    floor_mask = cv2.imread(floor_image_path)
    floor_mask = cv2.resize(floor_mask, (width, height), interpolation=cv2.INTER_LINEAR)

    # Every floor pixel, black out the canny image
    # We don't want to keep the canny lines on the floor
    canny_image_path = canny(image)
    canny_orig = cv2.imread(canny_image_path)
    canny_orig[np.where(np.all(floor_mask == 255, axis=-1))] = (0,0,0)
    cv2.imwrite(canny_image_path, canny_orig)

    if style_image is None:
        pipeline = "sdxl_t2i_adapter"
        args = {
            "prompt": prompt,
            "adapter_types": ["depth", "canny"],
            "adapter_images_conditioned": [depth_image_path, canny_image_path],
            "adapter_scales": [min(fidelity,1.0)**0.8, fidelity*0.4],
            "batch_size": batch_size,
            "steps": 25,
            "guidance_scale": 7.5,
            "seed": seed,
            "negative_prompt": negative_prompt,
            "width": width,
            "height": height,
            "scheduler": "dpms",
        }
        subproc_util.subproc_launch(pipeline, job, fn=pipelines.pipeline_get_loader(pipeline))
        reimage.update_job(job, 'rendering')
        result = subproc.subproc_call(pipeline, t2i_adapter, args=args)
    else:
        pipeline = "sdxl_controlnet_ip_adapter"
        args = {
            "prompt": prompt,
            "control_types": ["depth-sdxl"],
            "control_images_conditioned": [depth_image_path],
            "control_scales": [fidelity*0.85],
            #"control_scales": [min(fidelity*1.4,1.0)**(1/5)],
            # "control_types": ["depth-sdxl", "canny-sdxl"],
            # "control_images_conditioned": [depth_image_path, canny_image_path],
            # "control_scales": [min(fidelity*1.4,1.0)**(1/5), fidelity*0.6],
            "ip_adapter_image": style_image,
            "ip_adapter_scale": fidelity,
            "batch_size": batch_size,
            "steps": 25,
            "guidance_scale": 7.5,
            "seed": seed,
            "negative_prompt": negative_prompt,
            "width": width,
            "height": height,
            "scheduler": "dpms",
        }
        subproc_util.subproc_launch(pipeline, job, fn=pipelines.pipeline_get_loader(pipeline))
        reimage.update_job(job, 'rendering')
        result = subproc.subproc_call(pipeline, controlnet, args=args)

    if isinstance(result, BaseException):
        raise result
    if isinstance(result, dict):
        return {**result, "extra-0-orig": image, "extra-1-style": style_image, "extra-2-canny": canny_image_path, "extra-3-floor": floor_image_path, "extra-4-depth": depth_image_path}
    return result


def remodel_interior_masked(
        job,
        batch_size,
        image,
        mask_image,
        style_image,
        prompt,
        negative_prompt,
        seed,
        fidelity,
        width,
        height,
):
    depth_image_path = depth(image)
    floor_image_path = seg(image, preset="floor", expand=-1)
    floor_mask = cv2.imread(floor_image_path)
    floor_mask = cv2.resize(floor_mask, (width, height), interpolation=cv2.INTER_LINEAR)

    # Every floor pixel, black out the canny image
    # We don't want to keep the canny lines on the floor
    canny_image_path = canny(image)
    canny_orig = cv2.imread(canny_image_path)
    canny_orig[np.where(np.all(floor_mask == 255, axis=-1))] = (0,0,0)
    cv2.imwrite(canny_image_path, canny_orig)

    if style_image is None:
        pipeline = "sdxl_inpainting_controlnet"
        args = {
            "prompt": prompt,
            "control_types": ["depth-sdxl", "canny-sdxl"],
            "control_scales": [fidelity, fidelity],
            "control_images_conditioned": [depth_image_path, canny_image_path],
            "batch_size": batch_size,
            "steps": 25,
            "guidance_scale": 10.0,
            "seed": seed,
            "negative_prompt": negative_prompt,
            "scheduler": "dpms",
            "input_image": image,
            "mask_image": mask_image,
            "overmask": True,
        }
        subproc_util.subproc_launch(pipeline, job, fn=pipelines.pipeline_get_loader(pipeline))
        reimage.update_job(job, 'rendering')
        result = subproc.subproc_call(pipeline, inpainting, args=args)
    else:
        pipeline = "sdxl_inpainting_controlnet_ip_adapter"
        args = {
            "prompt": prompt,
            "control_types": ["depth-sdxl"],
            "control_scales": [fidelity],
            "control_images_conditioned": [depth_image_path],
            "ip_adapter_image": style_image,
            "ip_adapter_scale": fidelity,
            "batch_size": batch_size,
            "steps": 25,
            "guidance_scale": 10.0,
            "strength": 0.999,
            "seed": seed,
            "negative_prompt": negative_prompt,
            "scheduler": "dpms",
            "input_image": image,
            "mask_image": mask_image,
            "overmask": True,
        }
        subproc_util.subproc_launch(pipeline, job, fn=pipelines.pipeline_get_loader(pipeline))
        reimage.update_job(job, 'rendering')
        result = subproc.subproc_call(pipeline, inpainting, args=args)

    if isinstance(result, BaseException):
        raise result
    if isinstance(result, dict):
        return {**result, "extra-0-orig": image, "extra-1-style": style_image, "extra-2-mask": mask_image, "extra-3-canny": canny_image_path, "extra-4-floor": floor_image_path, "extra-5-depth": depth_image_path}
    return result
