"""Test cases for comfy-service"""
# pylint: disable=bare-except
# pylint: disable=broad-exception-raised
import os
import pprint
import shutil
import re
import traceback
import sys
from pathlib import Path
from functools import partial
import math
import tempfile
from PIL import Image
import service

def filepath(value):
    """
    Checks to see if a str describes a file path and checks a few locations for convenience
    
    If so it returns the full file path.
    If no file was found returns None
    """
    # if not isinstance(value, Path):
    #     raise ValueError(f"Invalid path object {value}")
    for possible_file_path in [
            value,
            f"../{value}",
            f"{os.path.dirname(__file__)}/{value}",
            f"{os.path.dirname(__file__)}/../{value}",
            f"{os.path.dirname(__file__)}/../../{value}",
    ]:
        if os.path.isfile(possible_file_path):
            return Path(possible_file_path)
    raise Exception(f"Failed to find file {value}")

class TestCase:
    '''
    An individual test case

    name - a name for the test case
    inputs - a set of inputs
    success_fn - a success/failure evaluator for this test
    exception_expected - set to true if this test case will throw an exception
    '''
    def __init__(
            self,
            inputs,
            success_fn=None,
            exception_expected=False,
    ):
        self.inputs = inputs
        self.success_fn = lambda a: True
        self.exception_expected = exception_expected
        if success_fn is None:
            self.success_fn = lambda a: True
        else:
            self.success_fn = success_fn

        # Remove all None values from the inputs
        # In these tests "None" is a way to test unspecified
        # There is no way for a user to specify the literal None value from the calling the API
        for k in list(self.inputs.keys()):
            if self.inputs[k] is None:
                self.inputs.pop(k)


    def savename(self, count, outputkey, extension):
        '''get the a decent savename prefix'''
        copy = self.inputs.copy()
        copy.pop('op', None)
        if 'prompt' in copy:
            copy['prompt'] = 'prompt'
        if 'seed' in copy:
            copy['seed'] = f"{copy['seed']:04d}"
        if 'mask_image' in copy:
            copy['mask_image'] = os.path.basename(self.inputs.get('mask_image')).split(".")[0]
        if 'style_image' in copy:
            copy['style_image'] = os.path.basename(self.inputs.get('style_image')).split(".")[0]

        description = ""
        for name,val in copy.items():
            if name == "image":
                continue
            strval = str(val)
            strval = re.sub(r"\s+", "", strval)
            strval = re.sub(r"/", "_", strval)
            strval = strval if len(strval) <= 10 else f"{strval:10}"
            description = f"{name}:{strval}-{description}"
        description = description[:-1]
        image_base = os.path.basename(self.inputs.get('image')).split(".")[0]
        return f"output-{count:04d}-{self.inputs.get('op')}-{image_base}-{description}.{outputkey}{extension}"


    def run(self, savedir, count, extra_success_fn=None):
        '''run a test case'''
        print("Inputs:")
        print(f" {pprint.pformat(self.inputs, compact=False)[1:-1]}")
        try:
            inputs = self.inputs.copy()
            inputs['job'] = None
            outputs = service.run_inputs(inputs)
        except Exception as exc:
            if self.exception_expected:
                print(f"Exception was expected: {exc}")
                return True, {}
            raise exc

        # Save outputs
        for outputkey, outputvalue in outputs.items():
            if not isinstance(outputvalue, str):
                continue
            if not os.path.exists(outputvalue):
                continue
            if os.stat(outputvalue).st_size == 0:
                continue # skip if empty
            outputname = savedir + "/" + self.savename(count, outputkey, os.path.splitext(outputvalue)[1])
            print(f"Saving {outputkey} to {outputname}")
            shutil.copy(outputvalue, outputname)

        # Validate output
        result = self.success_fn(outputs)
        if result is True:
            result = extra_success_fn(outputs)
        return result, outputs


class TestCaseSet:
    '''
    Test Case defines a set of inputs and eval to check that the evaluated correctly
    
    success_fn - a success fn to check all test case results
    '''
    # pylint: disable=too-many-arguments
    def __init__(
            self,
            name,
            success_fn=None,
    ):
        self.name = name
        self.test_cases = []
        if success_fn is None:
            self.success_fn = lambda a: True
        else:
            self.success_fn = success_fn

    def append(self, case):
        '''append a inputs to the test set'''
        self.test_cases.append(case)

    def run(self, quick=False):
        '''run the set of test cases'''
        if len(self.test_cases) == 0:
            print(f"Skipping {self.name} (no test cases)")
            return True
        print(f"Running {self.name} (cases: {len(self.test_cases)})...")
        savedir = f"./outputs/outputs-{self.name}"
        os.makedirs(savedir, exist_ok=True)

        for i, testcase in enumerate(self.test_cases):
            testname = f"{self.name}-{i:04d}"
            print(f"==== Test {self.name} {testname} RUN  ====")

            try:
                result, outputs = testcase.run(savedir, i, extra_success_fn=self.success_fn)
            except:
                traceback.print_exc(file=sys.stdout)
                result, outputs = False, {}

            if not result:
                print(f"==== Test {self.name} {testname} FAIL ====")
                print("Inputs:")
                print(f" {pprint.pformat(testcase.inputs, compact=False)[1:-1]}")
                print("Outputs:")
                print(f" {pprint.pformat(outputs, compact=False)[1:-1]}")
                print(f"Result: {result}")
                return False, i+1 # immediately exit
            print(f"==== Test {self.name} {testname} PASS ====\n\n")
            if quick:
                return True, i+1
        # All tests passed
        return True, len(self.test_cases)

    @staticmethod
    def create_file_list(name, namefilter=None):
        '''creates an iterator for the files in a directory'''
        found_dir = None

        if os.path.isfile(os.path.join(os.path.dirname(__file__),f"../{name}")):
            return [os.path.join(os.path.dirname(__file__),f"../{name}")]
        if os.path.isfile(os.path.join(os.path.dirname(__file__),name)):
            return [os.path.join(os.path.dirname(__file__),name)]

        if os.path.exists(os.path.join(os.path.dirname(__file__),f"../{name}")):
            found_dir = os.path.join(os.path.dirname(__file__),f"../{name}")
        if os.path.exists(os.path.join(os.path.dirname(__file__),name)):
            found_dir = os.path.join(os.path.dirname(__file__),name)
        if found_dir is None:
            raise Exception(f"Unable to locate files directory: {name}")

        file_list = sorted(os.listdir(found_dir))
        file_list = [file for file in file_list if os.path.isfile(os.path.join(found_dir, file))]
        if namefilter is not None:
            file_list = [file for file in file_list if namefilter in file or namefilter.replace(" ","_") in file]
        file_list = [os.path.normpath(f"{found_dir}/{f}") for f in file_list]
        file_list = [Path(f) for f in file_list]
        return file_list


def _get_room_type_from_filename(filename, error_ok=False):
    '''deduces the room_type from filename'''
    room_types = ["hall", "kitchen", "living room", "bathroom", "bedroom", "dining room", "office", "garage", "laundry room"]
    for room_type in room_types:
        if room_type in str(filename):
            return room_type
        if room_type.replace(' ','_') in str(filename):
            return room_type
        if room_type.replace(' ','-') in str(filename):
            return room_type
    if error_ok:
        return None
    raise Exception(f"Unknown room_type for: {filename}")


def _auto_resize(image_path, target_pixel_count=1024*1024):
    """resize image to legal dimenions of aproximately 1 megapixel"""
    input_image = Image.open(image_path)
    orig_width, orig_height = input_image.size
    # print(f"original size: {orig_width} {orig_height}")

    if orig_width % 64 == 0 and orig_height % 64 == 0 and orig_width*orig_height < 1500000 and orig_width*orig_height > 500000:
        return image_path

    actual_pixel_count = orig_width * orig_height
    scale = math.sqrt(target_pixel_count/actual_pixel_count)
    # print(f"scale: {scale}")

    cropx = 0
    cropy = 0

    # evaluate two options
    # 1) scale the small side to the nearest 64 multiple below the target size, and crop extra
    # 2) scale the small side to the nearest 64 multiple above the target size, and crop extra
    # choose the option that requires the least amount of cropping
    if orig_width >= orig_height:
        upper_scaled_height = math.ceil((orig_height * scale) / 64.0) * 64
        upper_scaled_width = int((orig_width * (upper_scaled_height/orig_height)))
        upper_cropx = upper_scaled_width - math.floor(upper_scaled_width / 64.0) * 64
        lower_scaled_height = math.floor((orig_height * scale) / 64.0) * 64
        lower_scaled_width = int((orig_width * (lower_scaled_height/orig_height)))
        lower_cropx = lower_scaled_width - math.floor(lower_scaled_width / 64.0) * 64
        # print(f"option1: {upper_scaled_width}x{upper_scaled_height} with cropped pixels: {upper_cropx}")
        # print(f"option2: {lower_scaled_width}x{lower_scaled_height} with cropped pixels: {lower_cropx}")
        if lower_cropx <= upper_cropx:
            scaled_height = lower_scaled_height
            scaled_width = lower_scaled_width
            cropx = lower_cropx
        else:
            scaled_height = upper_scaled_height
            scaled_width = upper_scaled_width
            cropx = upper_cropx
    else:
        upper_scaled_width = math.ceil((orig_width * scale) / 64.0) * 64
        upper_scaled_height = int((orig_height * (upper_scaled_width/orig_width)))
        upper_cropy = upper_scaled_height - math.floor(upper_scaled_height / 64.0) * 64
        lower_scaled_width = math.floor((orig_width * scale) / 64.0) * 64
        lower_scaled_height = int((orig_height * (lower_scaled_width/orig_width)))
        lower_cropy = lower_scaled_height - math.floor(lower_scaled_height / 64.0) * 64
        # print(f"option1: {upper_scaled_width}x{upper_scaled_height} with cropped pixels: {upper_cropy}")
        # print(f"option2: {lower_scaled_width}x{lower_scaled_height} with cropped pixels: {lower_cropy}")
        if lower_cropy <= upper_cropy:
            scaled_height = lower_scaled_height
            scaled_width = lower_scaled_width
            cropy = lower_cropy
        else:
            scaled_height = upper_scaled_height
            scaled_width = upper_scaled_width
            cropy = upper_cropy

    # print(f"Choosing {scaled_width}x{scaled_height} cropx: {cropx} cropy: {cropy}")
    rescaled_image = input_image.resize((scaled_width, scaled_height), Image.LANCZOS)
    rescaled_width, rescaled_height = rescaled_image.size
    # print(f"rescaled size: {rescaled_width} {rescaled_height}")

    cropped_image = rescaled_image.crop((cropx/2, cropy/2, rescaled_width - cropx/2, rescaled_height - cropy/2))
    # cropped_width, cropped_height = cropped_image.size
    # print(f"cropped size: {cropped_width} {cropped_height}")
    resized_image_path = f"/tmp/{os.path.splitext(os.path.basename(image_path))[0]}-resized.png"
    cropped_image.save(resized_image_path,"PNG")
    print(f"Resized {image_path} -> {resized_image_path}")
    return resized_image_path


def interior_remodel_unmasked_sets(seed_list):
    '''create the test suite'''
    sets = []

    suite = TestCaseSet(
        "interior-remodel-preset-unmasked",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    presets = [
          "modern",
          "art deco",
          "asian zen",
          "bohemian",
          "cabin",
          "coastal interior",
          "contemporary",
          "cottage",
          "farmhouse",
          "french country",
          "industrial",
          "japandi",
          "mediterranean",
          "midcentury modern",
          "minimalist",
          "modern country",
          "new traditional",
          "organic modern",
          "post modern",
          "rustic",
          "scandinavian",
          "shabby chic",
          "southwestern",
          "traditional",
          "transitional",
          "tuscan",
    ]
    testcases = [
        (filepath("test-images/interior-remodel/bedroom_1.png")),
        (filepath("test-images/interior-remodel/bedroom_2.png")),
        (filepath("test-images/interior-remodel/dining_room_1.png")),
        (filepath("test-images/interior-remodel/kitchen_1.png")),
        (filepath("test-images/interior-remodel/kitchen_2.png")),
        (filepath("test-images/interior-remodel/kitchen_3.jpg")),
        (filepath("test-images/interior-remodel/living_room_1.png")),
        (filepath("test-images/interior-remodel/living_room_2.png")),
    ]
    for seed in seed_list:
        for preset in presets:
            for image in testcases:
                suite.append(TestCase({
                    "op": "interior-remodel",
                    "image": _auto_resize(image),
                    "preset": preset,
                    "room_type": _get_room_type_from_filename(image),
                    "seed": seed,
                }))
    sets.append(suite)


    suite = TestCaseSet(
        "interior-remodel-prompt-unmasked",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    testcases = [
        (filepath("test-images/interior-remodel/bedroom_1.png"), "a beautiful modern bedroom"),
        (filepath("test-images/interior-remodel/dining_room_1.png"), "a beautiful modern dining room"),
        (filepath("test-images/interior-remodel/kitchen_1.png"), "a beautiful modern kitchen"),
        (filepath("test-images/interior-remodel/living_room_1.png"), "a beautiful modern living room"),
    ]
    for seed in seed_list:
        for image, prompt in testcases:
            suite.append(TestCase({
                "op": "interior-remodel",
                "image": _auto_resize(image),
                "prompt": prompt,
                "seed": seed,
            }))
    sets.append(suite)

    suite = TestCaseSet(
        "interior-remodel-image-unmasked",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    testcases = [
        (filepath("test-images/interior-remodel/bedroom_1.png")),
        (filepath("test-images/interior-remodel/dining_room_1.png")),
        (filepath("test-images/interior-remodel/kitchen_1.png")),
        (filepath("test-images/interior-remodel/living_room_1.png")),
    ]
    for seed in seed_list:
        for image in testcases:
            room_type = _get_room_type_from_filename(image)
            for style_image in TestCaseSet.create_file_list("test-images/style-images/", namefilter=room_type):
                suite.append(TestCase({
                    "op": "interior-remodel",
                    "image": _auto_resize(image),
                    "style_image": style_image,
                    "seed": seed,
                }))
    sets.append(suite)

    return sets


def interior_remodel_masked_sets(seed_list):
    '''create the test suite'''
    sets = []

    suite = TestCaseSet(
        "interior-remodel-preset-masked",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    presets = [
          "modern",
          "art deco",
          "asian zen",
          "bohemian",
          "cabin",
          "coastal interior",
          "contemporary",
          "cottage",
          "farmhouse",
          "french country",
          "industrial",
          "japandi",
          "mediterranean",
          "midcentury modern",
          "minimalist",
          "modern country",
          "new traditional",
          "organic modern",
          "post modern",
          "rustic",
          "scandinavian",
          "shabby chic",
          "southwestern",
          "traditional",
          "transitional",
          "tuscan",
    ]
    masked_testcases = [
        (filepath("test-images/interior-remodel/bedroom_1.png"), filepath("test-images/interior-remodel/bedroom_1.mask.png")),
        (filepath("test-images/interior-remodel/bedroom_2.png"), filepath("test-images/interior-remodel/bedroom_2.mask.png")),
        (filepath("test-images/interior-remodel/dining_room_1.png"), filepath("test-images/interior-remodel/dining_room_1.mask.png")),
        (filepath("test-images/interior-remodel/kitchen_1.png"), filepath("test-images/interior-remodel/kitchen_1.mask.png")),
        (filepath("test-images/interior-remodel/kitchen_2.png"), filepath("test-images/interior-remodel/kitchen_2.mask.png")),
        (filepath("test-images/interior-remodel/living_room_1.png"), filepath("test-images/interior-remodel/living_room_1.mask.png")),
        (filepath("test-images/interior-remodel/living_room_2.png"), filepath("test-images/interior-remodel/living_room_2.mask.png")),
    ]
    for seed in seed_list:
        for preset in presets:
            for image, mask_image in masked_testcases:
                suite.append(TestCase({
                    "op": "interior-remodel",
                    "image": image,
                    "mask_image": mask_image,
                    "room_type": _get_room_type_from_filename(image),
                    "preset": preset,
                    "seed": seed,
                }))
    sets.append(suite)

    suite = TestCaseSet(
        "interior-remodel-prompt-masked",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    masked_testcases = [
        (filepath("test-images/interior-remodel/bedroom_1.png"), filepath("test-images/interior-remodel/bedroom_1.mask.png"), "a beautiful modern bedroom"),
        (filepath("test-images/interior-remodel/bedroom_2.png"), filepath("test-images/interior-remodel/bedroom_2.mask.png"), "a beautiful modern bedroom"),
        (filepath("test-images/interior-remodel/dining_room_1.png"), filepath("test-images/interior-remodel/dining_room_1.mask.png"), "a beautiful modern dining room"),
        (filepath("test-images/interior-remodel/kitchen_1.png"), filepath("test-images/interior-remodel/kitchen_1.mask.png"), "a beautiful modern kitchen"),
        (filepath("test-images/interior-remodel/kitchen_2.png"), filepath("test-images/interior-remodel/kitchen_2.mask.png"), "a beautiful modern kitchen"),
        (filepath("test-images/interior-remodel/living_room_1.png"), filepath("test-images/interior-remodel/living_room_1.mask.png"), "a beautiful modern living room"),
        (filepath("test-images/interior-remodel/living_room_2.png"), filepath("test-images/interior-remodel/living_room_2.mask.png"), "a beautiful modern living room"),
    ]
    for seed in seed_list:
        for image, mask_image, prompt in masked_testcases:
            suite.append(TestCase({
                "op": "interior-remodel",
                "image": image,
                "mask_image": mask_image,
                "prompt": prompt,
                "seed": seed,
            }))
    sets.append(suite)

    suite = TestCaseSet(
        "interior-remodel-image-masked",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    masked_testcases = [
        (filepath("test-images/interior-remodel/bedroom_1.png"), filepath("test-images/interior-remodel/bedroom_1.mask.png")),
        (filepath("test-images/interior-remodel/bedroom_2.png"), filepath("test-images/interior-remodel/bedroom_2.mask.png")),
        (filepath("test-images/interior-remodel/dining_room_1.png"), filepath("test-images/interior-remodel/dining_room_1.mask.png")),
        (filepath("test-images/interior-remodel/kitchen_1.png"), filepath("test-images/interior-remodel/kitchen_1.mask.png")),
        (filepath("test-images/interior-remodel/kitchen_2.png"), filepath("test-images/interior-remodel/kitchen_2.mask.png")),
        (filepath("test-images/interior-remodel/living_room_1.png"), filepath("test-images/interior-remodel/living_room_1.mask.png")),
        (filepath("test-images/interior-remodel/living_room_2.png"), filepath("test-images/interior-remodel/living_room_2.mask.png")),
    ]
    for seed in seed_list:
        for image, mask_image in masked_testcases:
            room_type = _get_room_type_from_filename(image)
            for style_image in TestCaseSet.create_file_list("test-images/style-images/", namefilter=room_type):
                suite.append(TestCase({
                    "op": "interior-remodel",
                    "image": image,
                    "style_image": style_image,
                    "room_type": _get_room_type_from_filename(image),
                    "seed": seed,
                }))
    sets.append(suite)

    return sets


def exterior_remodel_unmasked_sets(seed_list):
    '''create the test suite'''
    sets = []

    suite = TestCaseSet(
        "exterior-remodel-preset-unmasked",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    presets = [
        "american craftsman",
        "art deco",
        "art nuveau",
        "bahaus",
        "brutalist",
        "bungalow",
        "cabin",
        "cape cod",
        "colonial",
        "cottage",
        "french provential",
        "georgian",
        "gothic",
        "greek",
        "mediterranean",
        "mid century",
        "modern",
        "monterey",
        "mountain modern",
        "national",
        "northwest regional style",
        "queen anne",
        "ranch style",
        "santa fe",
        "traditional",
        "tudor",
        "tuscan",
        "victorian",
    ]
    testcases = [
        (filepath("test-images/exterior/01-exterior-1024x1024.jpg")),
        (filepath("test-images/exterior/02-exterior-1216x896.jpg")),
        (filepath("test-images/exterior/03-exterior-1216x960.jpg")),
        (filepath("test-images/exterior/05-exterior-1216x832.jpg")),
    ]
    for seed in seed_list:
        for preset in presets:
            for image in testcases:
                suite.append(TestCase({
                    "op": "exterior-remodel",
                    "image": image,
                    "preset": preset,
                    "structure_type": "house",
                    "seed": seed,
                }))
    sets.append(suite)


    suite = TestCaseSet(
        "exterior-remodel-prompt-unmasked",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    testcases = [
        (filepath("test-images/exterior/01-exterior-1024x1024.jpg"), "a beautiful modern house"),
        (filepath("test-images/exterior/02-exterior-1216x896.jpg"), "a beautiful modern house"),
        (filepath("test-images/exterior/03-exterior-1216x960.jpg"), "a beautiful modern house"),
        (filepath("test-images/exterior/05-exterior-1216x832.jpg"), "a beautiful modern house"),
    ]
    for seed in seed_list:
        for image, prompt in testcases:
            suite.append(TestCase({
                "op": "exterior-remodel",
                "image": image,
                "prompt": prompt,
                "seed": seed,
            }))
    sets.append(suite)

    suite = TestCaseSet(
        "exterior-remodel-image-unmasked",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    testcases = [
        (filepath("test-images/exterior/01-exterior-1024x1024.jpg")),
        (filepath("test-images/exterior/02-exterior-1216x896.jpg")),
        (filepath("test-images/exterior/03-exterior-1216x960.jpg")),
        (filepath("test-images/exterior/05-exterior-1216x832.jpg")),
    ]
    for seed in seed_list:
        for image in testcases:
            for style_image in TestCaseSet.create_file_list("test-images/style-images/", namefilter="exterior"):
                suite.append(TestCase({
                    "op": "exterior-remodel",
                    "image": image,
                    "style_image": style_image,
                    "seed": seed,
                }))
    sets.append(suite)

    return sets


def exterior_remodel_masked_sets(seed_list):
    '''create the test suite'''
    sets = []

    suite = TestCaseSet(
        "exterior-remodel-preset-masked",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    presets = [
        "american craftsman",
        "art deco",
        "art nuveau",
        "bahaus",
        "brutalist",
        "bungalow",
        "cabin",
        "cape cod",
        "colonial",
        "cottage",
        "french provential",
        "georgian",
        "gothic",
        "greek",
        "mediterranean",
        "mid century",
        "modern",
        "monterey",
        "mountain modern",
        "national",
        "northwest regional style",
        "queen anne",
        "ranch style",
        "santa fe",
        "traditional",
        "tudor",
        "tuscan",
        "victorian",
    ]
    testcases = [
        (filepath("test-images/exterior/01-exterior-1024x1024.jpg"), filepath("test-images/exterior/01-exterior-1024x1024-mask.png")),
        (filepath("test-images/exterior/02-exterior-1216x896.jpg"), filepath("test-images/exterior/02-exterior-1216x896-mask.png")),
        (filepath("test-images/exterior/03-exterior-1216x960.jpg"), filepath("test-images/exterior/03-exterior-1216x960-mask.png")),
        (filepath("test-images/exterior/05-exterior-1216x832.jpg"), filepath("test-images/exterior/05-exterior-1216x832-mask.png")),
     ]
    for seed in seed_list:
        for preset in presets:
            for image, mask_image in testcases:
                suite.append(TestCase({
                    "op": "exterior-remodel",
                    "image": image,
                    "mask_image": mask_image,
                    "preset": preset,
                    "structure_type": "house",
                    "seed": seed,
                }))
    sets.append(suite)


    suite = TestCaseSet(
        "exterior-remodel-prompt-unmasked",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    testcases = [
        (filepath("test-images/exterior/01-exterior-1024x1024.jpg"), filepath("test-images/exterior/01-exterior-1024x1024-mask.png"), "a beautiful modern house"),
        (filepath("test-images/exterior/02-exterior-1216x896.jpg"), filepath("test-images/exterior/02-exterior-1216x896-mask.png"), "a beautiful modern house"),
        (filepath("test-images/exterior/03-exterior-1216x960.jpg"), filepath("test-images/exterior/03-exterior-1216x960-mask.png"), "a beautiful modern house"),
        (filepath("test-images/exterior/05-exterior-1216x832.jpg"), filepath("test-images/exterior/05-exterior-1216x832-mask.png"), "a beautiful modern house"),
    ]
    for seed in seed_list:
        for image, mask_image, prompt in testcases:
            suite.append(TestCase({
                "op": "exterior-remodel",
                "image": image,
                "mask_image": mask_image,
                "prompt": prompt,
                "seed": seed,
            }))
    sets.append(suite)

    suite = TestCaseSet(
        "exterior-remodel-image-unmasked",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    testcases = [
        (filepath("test-images/exterior/01-exterior-1024x1024.jpg"), filepath("test-images/exterior/01-exterior-1024x1024-mask.png")),
        (filepath("test-images/exterior/02-exterior-1216x896.jpg"), filepath("test-images/exterior/02-exterior-1216x896-mask.png")),
        (filepath("test-images/exterior/03-exterior-1216x960.jpg"), filepath("test-images/exterior/03-exterior-1216x960-mask.png")),
        (filepath("test-images/exterior/05-exterior-1216x832.jpg"), filepath("test-images/exterior/05-exterior-1216x832-mask.png")),
    ]
    for seed in seed_list:
        for image, mask_image in testcases:
            for style_image in TestCaseSet.create_file_list("test-images/style-images/", namefilter="exterior"):
                suite.append(TestCase({
                    "op": "exterior-remodel",
                    "image": _auto_resize(image),
                    "mask_image": mask_image,
                    "style_image": style_image,
                    "seed": seed,
                }))
    sets.append(suite)

    return sets


def classify_room_sets(seed_list):
    '''create the test suite'''
    sets = []

    suite = TestCaseSet(
        "classify-room",
        success_fn=lambda outputs : True,
    )

    # no seed input for classify room
    for i, image in enumerate(TestCaseSet.create_file_list("test-images/classify-room/")):
        if i > 9:
            break
        room_type = _get_room_type_from_filename(image)
        def check_room_type(outputs, room_type):
            print(f"Checking {outputs.get('room_type')} for {room_type}")
            return room_type in str(outputs.get('room_type'))

        testcase = TestCase(
            {
                "op": "classify-room",
                # "image": _auto_resize(image),
                "image": image,
            },
            success_fn = partial(check_room_type, room_type=room_type),
        )
        suite.append(testcase)

    sets.append(suite)

    return sets


def segmentation_sets(seed_list):
    '''create the test suite'''
    sets = []

    suite = TestCaseSet(
        "seg-presets-interior",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    presets = [
        "wall",
        "cabinet",
        "door",
        "floor",
    ]
    for preset in presets:
        suite.append(TestCase({
            "op": "seg",
            "image": filepath("test-images/interior-remodel/kitchen_1.png"),
            "preset": preset,
        }))
    sets.append(suite)

    suite = TestCaseSet(
        "seg-presets-exterior",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    presets = [
        "landscaping",
        "outside-wall",
        "brick",
        "shutter",
    ]
    for preset in presets:
        suite.append(TestCase({
            "op": "seg",
            "image": filepath("test-images/exterior/01-exterior-1024x1024.jpg"),
            "preset": preset,
        }))
    sets.append(suite)


    suite = TestCaseSet(
        "seg-presets-points",
        success_fn=lambda outputs : outputs.get('image-0') is not None,
    )
    suite.append(TestCase({
        "op": "seg",
        "image": filepath("test-images/kitchen/01-kitchen-1024x1024.jpg"),
        "points": [[859, 336]],
    }))
    suite.append(TestCase({
        "op": "seg",
        "image": filepath("test-images/kitchen/01-kitchen-1024x1024.jpg"),
        "points": [[134, 656], [158, 656], [180, 654], [204, 651], [228, 646], [249, 642], [266, 635], [282, 625], [300, 617], [319, 622]],
    }))
    sets.append(suite)

    return sets

def create_sets(seed_list):
    """create all test sets"""
    sets = [
        *interior_remodel_unmasked_sets(seed_list),
        *interior_remodel_masked_sets(seed_list),
        *exterior_remodel_unmasked_sets(seed_list),
        *exterior_remodel_masked_sets(seed_list),
        *classify_room_sets(seed_list),
        *segmentation_sets(seed_list),
    ]
    return sets
