import cv2
import uuid
import math
from PIL import Image
import tempfile

def tmpname(prefix="tmpfile", suffix=".tmp"):
    random=uuid.uuid4().hex[:8]
    return f"/tmp/{prefix}{random}{suffix}"

def check_image_sizes(**kwargs):
    """
    verifies that all images passed are the same size and SD compatible
    if not, an description exception will be thrown
    return the width, height of all images
    """
    if len(kwargs) < 1:
        raise Exception("image required")
    cvimg = cv2.imread(kwargs["image"])
    if cvimg is None:
        raise Exception("bad image format: image")
    height, width, _ = cvimg.shape
    for name, img in kwargs.items():
        if name == "image":
            # already checked the image
            continue
        if name == "style_image":
            # style_image can be any size and need not match image
            continue
        if img is None:
            continue
        cvimg = cv2.imread(img)
        if cvimg is None:
            raise Exception(f"bad image format: {name}")
        h, w, _ = cvimg.shape
        if h % 64 != 0 or w % 64 != 0:
            raise Exception(f"image dimensions unsupported: {w}x{h}")
        if h != height or w != width:
            raise Exception(f"image dimensions differ: {width}x{height} {w}x{h}")
    return (width, height)

def auto_resize(image_path, target_pixel_count=1024*1024):
    """resize image to legal dimenions of aproximately 1 megapixel"""
    input_image = Image.open(image_path)
    orig_width, orig_height = input_image.size
    print(f"original size: {orig_width} {orig_height}")

    actual_pixel_count = orig_width * orig_height
    scale = math.sqrt(target_pixel_count/actual_pixel_count)
    print(f"scale: {scale}")

    cropx = 0
    cropy = 0

    # evaluate two options
    # 1) scale the small side to the nearest 64 multiple below the target size, and crop extra
    # 2) scale the small side to the nearest 64 multiple above the target size, and crop extra
    # choose the option that requires the least amount of cropping
    if orig_width >= orig_height:
        upper_scaled_height = math.ceil((orig_height * scale) / 64.0) * 64
        upper_scaled_width = int((orig_width * (upper_scaled_height/orig_height)))
        upper_cropx = upper_scaled_width - math.floor(upper_scaled_width / 64.0) * 64
        lower_scaled_height = math.floor((orig_height * scale) / 64.0) * 64
        lower_scaled_width = int((orig_width * (lower_scaled_height/orig_height)))
        lower_cropx = lower_scaled_width - math.floor(lower_scaled_width / 64.0) * 64
        print(f"option1: {upper_scaled_width}x{upper_scaled_height} with cropped pixels: {upper_cropx}")
        print(f"option2: {lower_scaled_width}x{lower_scaled_height} with cropped pixels: {lower_cropx}")
        if lower_cropx <= upper_cropx:
            scaled_height = lower_scaled_height
            scaled_width = lower_scaled_width
            cropx = lower_cropx
        else:
            scaled_height = upper_scaled_height
            scaled_width = upper_scaled_width
            cropx = upper_cropx
    else:
        upper_scaled_width = math.ceil((orig_width * scale) / 64.0) * 64
        upper_scaled_height = int((orig_height * (upper_scaled_width/orig_width)))
        upper_cropy = upper_scaled_height - math.floor(upper_scaled_height / 64.0) * 64
        lower_scaled_width = math.floor((orig_width * scale) / 64.0) * 64
        lower_scaled_height = int((orig_height * (lower_scaled_width/orig_width)))
        lower_cropy = lower_scaled_height - math.floor(lower_scaled_height / 64.0) * 64
        print(f"option1: {upper_scaled_width}x{upper_scaled_height} with cropped pixels: {upper_cropy}")
        print(f"option2: {lower_scaled_width}x{lower_scaled_height} with cropped pixels: {lower_cropy}")
        if lower_cropy <= upper_cropy:
            scaled_height = lower_scaled_height
            scaled_width = lower_scaled_width
            cropy = lower_cropy
        else:
            scaled_height = upper_scaled_height
            scaled_width = upper_scaled_width
            cropy = upper_cropy

    print(f"Choosing {scaled_width}x{scaled_height} cropx: {cropx} cropy: {cropy}")
    rescaled_image = input_image.resize((scaled_width, scaled_height), Image.LANCZOS)
    rescaled_width, rescaled_height = rescaled_image.size
    print(f"rescaled size: {rescaled_width} {rescaled_height}")

    cropped_image = rescaled_image.crop((cropx/2, cropy/2, rescaled_width - cropx/2, rescaled_height - cropy/2))
    cropped_width, cropped_height = cropped_image.size
    print(f"cropped size: {cropped_width} {cropped_height}")
    _, resized_image_path = tempfile.mkstemp(suffix=".png",prefix="reimage-")
    cropped_image.save(resized_image_path,"PNG")
    return resized_image_path
