"""reimage service helper functions"""
import os
import sys
import time
import datetime
import json
import tempfile
import traceback
import requests
import urllib
from dateutil.parser import parse as parsedate

BASE_URL = ""
TIMEOUT = 5
VERIFY=True
active_job_id = None

def get_job(job_name, priority, timeout_sec):
    """get a job"""
    try:
        resp = requests.get(f"{BASE_URL}/job/{job_name}/wait?priority={priority}&timeout={timeout_sec*1000}", verify=VERIFY)
        if resp.status_code not in [200, 204]:
            print(f"get_job: Invalid response from broker: {resp.status_code}", flush=True)
            time.sleep(timeout_sec)
            return None
        if resp == "" or resp is None or resp.status_code == 204:
            return None
        return resp.json()
    except:
        traceback.print_exc()
        time.sleep(timeout_sec)
        return None


def update_job(job_id, status, results=None, detail=None, percent=None):
    """update job"""
    js = {}
    js['jobID'] = job_id
    js['status'] = status
    if detail is not None:
        js['detail'] = detail
    if percent is not None:
        js['percent'] = percent
    open_files = []
    try:
        multipart_form_data = {
            'json': (None, json.dumps(js)),
        }
        if results != {} and results is not None:
            for key, value in results.items():
                f = open(value, 'rb')
                open_files.append(f)
                multipart_form_data[key] = (key, f)

        r = requests.post(f"{BASE_URL}/job/{job_id}/update", files=multipart_form_data, verify=VERIFY)

        if r.status_code != 200:
            print(f"update_job: Invalid response from broker: {r.status_code}", flush=True)
    except:
        traceback.print_exc()
    finally:
        for f in open_files:
            f.close()


def update_job_error(job_id, error_str):
    """update job with the specified error string"""
    print(error_str, flush=True)
    update_job(job_id, "error", detail=f"error: {error_str}")


def requeue_active_job():
    """requeue a job for another worker to handle"""
    global active_job_id
    if active_job_id is None:
        return
    try:
        resp = requests.post(f"{BASE_URL}/job/{active_job_id}/requeue", verify=VERIFY)
        if resp.status_code not in [200, 204]:
            print(f"requeue: Invalid response from broker: {resp.status_code}", flush=True)
        return
    except:
        traceback.print_exc()
        return


def notify(message):
    """just sends a message for logging"""
    try:
        message = urllib.parse.quote(message)
        resp = requests.post(f"{BASE_URL}/notify?message={message}", verify=VERIFY)
        if resp.status_code not in [200, 204]:
            print(f"notify: Invalid response from broker: {resp.status_code}", flush=True)
        return
    except:
        traceback.print_exc()
        return


def get_input(input_name):
    url = f"{BASE_URL}{input_name}"
    fd, path = tempfile.mkstemp(suffix="",prefix="reimage-")
    with os.fdopen(fd, 'wb') as tmpo:
        resp = requests.get(url, verify=VERIFY)
        tmpo.write(resp.content)
    return path


def get_file(filepath):
    try:
        r = requests.head(f"{BASE_URL}{filepath}", verify=VERIFY)
        if r.status_code != 200:
            print(f"Failed to fetch file {filepath} {r.status_code}", flush=True)
            sys.exit(1)
        url_date = parsedate(r.headers['Last-Modified']).astimezone()
        file_time = datetime.datetime.fromtimestamp(0).astimezone()
        if os.path.exists(filepath):
            file_time = datetime.datetime.fromtimestamp(os.path.getmtime(filepath)).astimezone()
        os.makedirs(os.path.split(filepath)[0], exist_ok=True)
        if url_date > file_time:
            print(f"Downloading {filepath}:", end="")
            with requests.get(f"{BASE_URL}{filepath}", verify=VERIFY, stream=True) as r:
                r.raise_for_status()
                i = 0
                path, _ = os.path.split(filepath)
                os.makedirs(path, exist_ok=True)
                with open(filepath, 'wb+') as f:
                    i = i+1
                    if i%100 == 0:
                        print(".", end="")
                    for chunk in r.iter_content(chunk_size=8192):
                        f.write(chunk)
            print("")
    except:
        traceback.print_exc()
        sys.exit(1)


def dojob(job_name, process, priority, timeout):
    global active_job_id

    job_json = get_job(job_name, priority, timeout)
    if job_json is None:
        return
    job_id = job_json.get('jobID')
    if job_json.get('jobID') is None:
        print("Invalid job: jobID", flush=True)
        return

    active_job_id = job_id
    inputs = {}
    try:
        for k, v in job_json.get('inputs').items():
            inputs[k] = get_input(v)
        params = job_json.get('params') if job_json.get('params') is not None else {}

        process(job_id, **inputs, **params)
    except:
        traceback.print_exc()
        update_job(job_id, 'error')
    finally:
        active_job_id = None
        for _, v in inputs.items():
            os.remove(v)
